% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/baseline_wrappers.R
\name{baseline_gaussian}
\alias{baseline_gaussian}
\title{Create baseline evaluations for regression models}
\usage{
baseline_gaussian(
  test_data,
  train_data,
  dependent_col,
  n = 100,
  metrics = list(),
  random_effects = NULL,
  min_training_rows = 5,
  min_training_rows_left_out = 3,
  REML = FALSE,
  parallel = FALSE
)
}
\arguments{
\item{test_data}{Data frame.}

\item{train_data}{Data frame.}

\item{dependent_col}{Name of dependent variable in the supplied test and training sets.}

\item{n}{The number of random samplings of \code{train_data} to fit baseline models on. (Default is \code{100})}

\item{metrics}{List for enabling/disabling metrics.

  E.g. \code{list("RMSE" = FALSE)} would remove \code{RMSE} from the results,
  and \code{list("TAE" = TRUE)} would add the \code{Total Absolute Error} metric
  to the results.
  Default values (\code{TRUE}/\code{FALSE}) will be used for the remaining available metrics.

  You can enable/disable all metrics at once by including
  \code{"all" = TRUE/FALSE} in the list. This is done prior to enabling/disabling
  individual metrics, why f.i. \code{list("all" = FALSE, "RMSE" = TRUE)}
  would return only the \code{RMSE} metric.

  The list can be created with
  \code{\link[cvms:gaussian_metrics]{gaussian_metrics()}}.

  Also accepts the string \code{"all"}.}

\item{random_effects}{Random effects structure for baseline model. (Character)

 E.g. with \code{"(1|ID)"}, the model becomes \code{"y ~ 1 + (1|ID)"}.}

\item{min_training_rows}{Minimum number of rows in the random subsets of \code{train_data}.}

\item{min_training_rows_left_out}{Minimum number of rows left out of the random subsets of \code{train_data}.

 I.e. a subset will maximally have the size:

 \code{max_rows_in_subset = nrow(train_data) - min_training_rows_left_out}.}

\item{REML}{Whether to use Restricted Maximum Likelihood. (Logical)}

\item{parallel}{Whether to run the \code{n} evaluations in parallel. (Logical)

 Remember to register a parallel backend first.
 E.g. with \code{doParallel::registerDoParallel}.}
}
\value{
List containing:

 \enumerate{
  \item a tibble with summarized results (called \code{summarized_metrics})
  \item a tibble with random evaluations (\code{random_evaluations})
 }

 ....................................................................

 The \strong{Summarized Results} tibble contains:

 Average \strong{\code{RMSE}}, \strong{\code{MAE}}, \strong{\code{NRMSE(IQR)}},
 \strong{\code{RRSE}}, \strong{\code{RAE}}, \strong{\code{RMSLE}}.

 See the additional metrics (disabled by default) at \code{\link[cvms:gaussian_metrics]{?gaussian_metrics}}.

 The \strong{Measure} column indicates the statistical descriptor used on the evaluations.
 The row where \code{Measure == All_rows} is the evaluation when the baseline model
 is trained on all rows in \code{train_data}.

 The \strong{Training Rows} column contains the aggregated number of rows used from \code{train_data},
 when fitting the baseline models.

 ....................................................................

 The \strong{Random Evaluations} tibble contains:

 The \strong{non-aggregated metrics}.

 A nested tibble with the \strong{predictions} and targets.

 A nested tibble with the \strong{coefficients} of the baseline models.

 Number of \strong{training rows} used when fitting the baseline model on the training set.

 Specified \strong{family}.

 Name of \strong{dependent} variable.

 Name of \strong{fixed} effect (bias term only).

 \strong{Random} effects structure (if specified).
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("maturing")}

 Create a baseline evaluation of a test set.

 In modelling, a \code{baseline} is a result that
 is meaningful to compare the results from our models to. In regression, we
 want our model to be better than a model without any predictors. If our
 model does not perform better than such a simple model, it's unlikely to
 be useful.

 \code{baseline_gaussian()} fits the intercept-only model (\code{y ~ 1}) on \code{n} random
 subsets of \code{train_data} and evaluates each model on \code{test_data}. Additionally, it evaluates a
 model fitted on all rows in \code{train_data}.
}
\details{
Packages used:

 \subsection{Models}{

 \code{\link[stats:lm]{stats::lm}}, \code{\link[lme4:lmer]{lme4::lmer}}
 }
 \subsection{Results}{

 r2m : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 r2c : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 AIC : \code{\link[stats:AIC]{stats::AIC}}

 AICc : \code{\link[MuMIn:AICc]{MuMIn::AICc}}

 BIC : \code{\link[stats:BIC]{stats::BIC}}

 }
}
\examples{
\donttest{
# Attach packages
library(cvms)
library(groupdata2) # partition()
library(dplyr) # \%>\% arrange()

# Data is part of cvms
data <- participant.scores

# Set seed for reproducibility
set.seed(1)

# Partition data
partitions <- partition(data, p = 0.7, list_out = TRUE)
train_set <- partitions[[1]]
test_set <- partitions[[2]]

# Create baseline evaluations
# Note: usually n=100 is a good setting

baseline_gaussian(
  test_data = test_set,
  train_data = train_set,
  dependent_col = "score",
  random_effects = "(1|session)",
  n = 2
)

# Parallelize evaluations

# Attach doParallel and register four cores
# Uncomment:
# library(doParallel)
# registerDoParallel(4)

# Make sure to uncomment the parallel argument
baseline_gaussian(
  test_data = test_set,
  train_data = train_set,
  dependent_col = "score",
  random_effects = "(1|session)",
  n = 4
  #, parallel = TRUE  # Uncomment
)
}
}
\seealso{
Other baseline functions: 
\code{\link{baseline_binomial}()},
\code{\link{baseline_multinomial}()},
\code{\link{baseline}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{baseline functions}
