% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg_gds.R
\name{coxmeg_gds}
\alias{coxmeg_gds}
\title{Perform GWAS using a Cox mixed-effects model with GDS files as input}
\usage{
coxmeg_gds(
  gds,
  pheno,
  corr,
  type,
  cov = NULL,
  tau = NULL,
  snp.id = NULL,
  maf = 0.05,
  min_tau = 1e-04,
  max_tau = 5,
  eps = 1e-06,
  order = NULL,
  detap = NULL,
  opt = "bobyqa",
  score = FALSE,
  threshold = 0,
  solver = NULL,
  spd = TRUE,
  mc = 100,
  verbose = TRUE
)
}
\arguments{
\item{gds}{A GDS object created by \code{\link{snpgdsOpen}} or \code{\link{seqOpen}}.}

\item{pheno}{A data.frame containing the following four columns, family ID, individual ID, time and status. The status is a binary variable (1 for events/0 for censored). Missing values should be denoted as NA.}

\item{corr}{A relatedness matrix. Can be a matrix or a 'dgCMatrix' class in the Matrix package. Must be symmetric positive definite or symmetric positive semidefinite.}

\item{type}{A string indicating the sparsity structure of the relatedness matrix. Should be 'bd' (block diagonal), 'sparse', or 'dense'. See details.}

\item{cov}{An optional data.frame of covariates. Same as \code{pheno}, the first two columns are family ID and individual ID. The covariates are included in the null model for estimating the variance component. The covariates can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Missing values should be denoted as NA.}

\item{tau}{An optional positive value for the variance component. If tau is given, the function will skip estimating the variance component, and use the given tau to analyze the SNPs.}

\item{snp.id}{An optional vector of snp.id (or variant.id). Only these SNPs will be analyzed. Default is \code{NULL}, which selects all SNPs.}

\item{maf}{An optional positive value. All SNPs with MAF<maf in the GDS file will not be analyzed. If \code{snp.id} is not \code{NULL}, both SNP filters will be applied in combination. Default is 0.05.}

\item{min_tau}{An optional positive value indicating the lower bound in the optimization algorithm for the variance component tau. Default is 1e-4.}

\item{max_tau}{An optional positive value indicating the upper bound in the optimization algorithm for the variance component tau. Default is 5.}

\item{eps}{An optional positive value indicating the relative convergence tolerance in the optimization algorithm. Default is 1e-6. A smaller value (e.g., 1e-8) can be used for better precision of the p-values in the situation where most SNPs under investigation have a very low minor allele count (<5).}

\item{order}{An optional integer value starting from 0. Only effective when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is NULL, which lets coxmeg choose an optimal order.}

\item{detap}{An optional string indicating whether to use approximation for log-determinant. Can be 'exact', 'diagonal' or 'slq'. Default is NULL, which lets the function select a method based on 'type' and other information. See details.}

\item{opt}{An optional string value for the Optimization algorithm for tau. Can have the following values: 'bobyqa', 'Brent' or 'NM'. Default is 'bobyqa'.}

\item{score}{An optional logical value indicating whether to perform a score test. Default is FALSE.}

\item{threshold}{An optional non-negative value. If threshold>0, coxmeg_gds will reestimate HRs for those SNPs with a p-value<threshold by first estimating a variant-specific variance component. Default is 0.}

\item{solver}{An optional binary value taking either 1 or 2. Default is 1. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{mc}{An optional integer value specifying the number of Monte Carlo samples used for approximating the log-determinant. Only valid when \code{dense=TRUE} and \code{detap='slq'}. Default is 100.}

\item{verbose}{An optional logical value indicating whether to print additional messages. Default is TRUE.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

p: The p-value of each SNP.

tau: The estimated variance component.

rank: The rank of the relatedness matrix.

nsam: Actual sample size.
}
\description{
\code{coxmeg_gds} first estimates the variance component under a null model with only cov if tau is not given, and then analyzing each SNP in the gds file.
}
\section{About \code{type}}{

'bd' is used for a block-diagonal relatedness matrix, or a sparse matrix the inverse of which is also sparse. 'sparse' is used for a general sparse relatedness matrix the inverse of which is not sparse.
}

\section{About \code{corr}}{

The subjects in \code{corr} must be in the same order as in \code{pheno} and \code{cov}.
}

\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, set \code{spd=FALSE}.
}

\section{About \code{solver}}{

When \code{solver=1,3}/\code{solver=2}, Cholesky decompositon/PCG is used to solve the linear system. When \code{solver=3}, the solve function in the Matrix package is used, and when \code{solver=1}, it uses RcppEigen:LDLT to solve linear systems. When \code{type='dense'}, it is recommended to set \code{solver=2} to have better computational performance.
}

\section{About \code{detap}}{

When \code{detap='exact'}, the exact log-determinant is computed for estimating the variance component. Specifying \code{detap='diagonal'} uses diagonal approximation, and is only effective for a sparse relatedness matrix. Specifying \code{detap='slq'} uses stochastic lanczos quadrature approximation.
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## build a block-diagonal relatedness matrix
n_f <- 600
mat_list <- list()
size <- rep(5,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))

## Example data files
pheno.file = system.file("extdata", "ex_pheno.txt", package = "coxmeg")
cov.file = system.file("extdata", "ex_cov.txt", package = "coxmeg")
bed = system.file("extdata", "example_null.bed", package = "coxmeg")
bed = substr(bed,1,nchar(bed)-4)

## Read phenotype and covariates
pheno <- read.table(pheno.file, header=FALSE, as.is=TRUE, na.strings="-9")
cov <- read.table(cov.file, header=FALSE, as.is=TRUE)

## SNPRelate GDS object
gdsfile <- tempfile()
SNPRelate::snpgdsBED2GDS(bed.fn=paste0(bed,".bed"), 
                         fam.fn=paste0(bed,".fam"), 
                         bim.fn=paste0(bed,".bim"), 
                         out.gdsfn=gdsfile, verbose=FALSE)
gds <- SNPRelate::snpgdsOpen(gdsfile)

## Estimate variance component under a null model
re <- coxmeg_gds(gds,pheno,sigma,type='bd',cov=cov,detap='diagonal',order=1)
re

SNPRelate::snpgdsClose(gds)
unlink(gdsfile)

## SeqArray GDS object
gdsfile <- tempfile()
SeqArray::seqBED2GDS(bed.fn=paste0(bed,".bed"), 
                     fam.fn=paste0(bed,".fam"), 
                     bim.fn=paste0(bed,".bim"), 
                     out.gdsfn=gdsfile, verbose=FALSE)
gds <- SeqArray::seqOpen(gdsfile)

## Estimate variance component under a null model
re <- coxmeg_gds(gds,pheno,sigma,type='bd',cov=cov,detap='diagonal',order=1)
re

SeqArray::seqClose(gds)
unlink(gdsfile)
}
\author{
Liang He, Stephanie Gogarten
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
