% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svyfgt.R
\name{svyfgt}
\alias{svyfgt}
\alias{svyfgt.survey.design}
\alias{svyfgt.svyrep.design}
\alias{svyfgt.DBIsvydesign}
\title{FGT measure of poverty}
\usage{
svyfgt(formula, design, ...)

\method{svyfgt}{survey.design}(
  formula,
  design,
  g,
  type_thresh = "abs",
  abs_thresh = NULL,
  percent = 0.6,
  quantiles = 0.5,
  na.rm = FALSE,
  thresh = FALSE,
  deff = FALSE,
  linearized = FALSE,
  influence = FALSE,
  ...
)

\method{svyfgt}{svyrep.design}(
  formula,
  design,
  g,
  type_thresh = "abs",
  abs_thresh = NULL,
  percent = 0.6,
  quantiles = 0.5,
  na.rm = FALSE,
  thresh = FALSE,
  deff = FALSE,
  linearized = FALSE,
  return.replicates = FALSE,
  ...
)

\method{svyfgt}{DBIsvydesign}(formula, design, ...)
}
\arguments{
\item{formula}{a formula specifying the income variable}

\item{design}{a design object of class \code{survey.design} or class \code{svyrep.design} from the \code{survey} library.}

\item{...}{passed to \code{svyarpr} and \code{svyarpt}}

\item{g}{If g=0 estimates the headcount ratio; If g=1 estimates the average normalised poverty gap, and if g=2 estimates the average squared normalised poverty gap}

\item{type_thresh}{type of poverty threshold. If "abs" the threshold is fixed and given the value
of abs_thresh; if "relq" it is given by percent times the quantile; if "relm" it is percent times the mean.}

\item{abs_thresh}{poverty threshold value if type_thresh is "abs"}

\item{percent}{the multiple of the the quantile or mean used in the poverty threshold definition}

\item{quantiles}{the quantile used used in the poverty threshold definition}

\item{na.rm}{Should cases with missing values be dropped?}

\item{thresh}{return the poverty threshold value}

\item{deff}{Return the design effect (see \code{survey::svymean})}

\item{linearized}{Should a matrix of linearized variables be returned?}

\item{influence}{Should a matrix of (weighted) influence functions be returned? (for compatibility with \code{\link[survey]{svyby}}). Not implemented yet for linearized designs.}

\item{return.replicates}{Return the replicate estimates?}
}
\value{
Object of class "\code{cvystat}", which are vectors with a "\code{var}" attribute giving the variance and a "\code{statistic}" attribute giving the name of the statistic.
}
\description{
Estimate the FGT measure.
}
\details{
you must run the \code{convey_prep} function on your survey design object immediately after creating it with the \code{svydesign} or \code{svrepdesign} function.

The FGT poverty measures have three special cases.
When \code{g = 0}, the FGT measure is the headcount poverty rate, assigning the same "poverty-weight" to all persons below the poverty line.
When \code{g = 1}, it becomes the poverty gap ratio, a measure which accounts for the intensity of income shortfall among the poor.
When \code{g = 2}. it becomes the squared poverty gap ratio, a measure that also accounts for inequality of poverty intesity across the poor.
The \code{g} is a poverty sensitivity parameter, adding more weight to people with greater income shortfalls as it increases.
}
\examples{
library(survey)
library(laeken)
data(eusilc) ; names( eusilc ) <- tolower( names( eusilc ) )

# linearized design

des_eusilc <- svydesign( ids = ~rb030 , strata = ~db040 ,  weights = ~rb050 , data = eusilc )
des_eusilc <- convey_prep( des_eusilc )

# replicate-weighted design
des_eusilc_rep <- as.svrepdesign( des_eusilc , type = "bootstrap" )
des_eusilc_rep <- convey_prep( des_eusilc_rep )

# headcount ratio, poverty threshold fixed
svyfgt(~eqincome, des_eusilc, g=0,  abs_thresh=10000)
# poverty gap index, poverty threshold fixed
svyfgt(~eqincome, des_eusilc, g=1,  abs_thresh=10000)
# headcount ratio, poverty threshold equal to arpt
svyfgt(~eqincome, des_eusilc, g=0, type_thresh= "relq" , thresh = TRUE)
# poverty gap index, poverty threshold equal to arpt
svyfgt(~eqincome, des_eusilc, g=1, type_thresh= "relq", thresh = TRUE)
# headcount ratio, poverty threshold equal to .6 times the mean
svyfgt(~eqincome, des_eusilc, g=0, type_thresh= "relm", thresh = TRUE)
# poverty gap index, poverty threshold equal to 0.6 times the mean
svyfgt(~eqincome, des_eusilc, g=1, type_thresh= "relm" , thresh = TRUE)

#  using svrep.design:
# headcount ratio, poverty threshold fixed
svyfgt(~eqincome, des_eusilc_rep, g=0,  abs_thresh=10000)
# poverty gap index, poverty threshold fixed
svyfgt(~eqincome, des_eusilc, g=1,  abs_thresh=10000)
# headcount ratio, poverty threshold equal to arpt
svyfgt(~eqincome, des_eusilc_rep, g=0, type_thresh= "relq" , thresh = TRUE)
# poverty gap index, poverty threshold equal to arpt
svyfgt(~eqincome, des_eusilc, g=1, type_thresh= "relq", thresh = TRUE)
# headcount ratio, poverty threshold equal to .6 times the mean
svyfgt(~eqincome, des_eusilc_rep, g=0, type_thresh= "relm" , thresh = TRUE)
# poverty gap index, poverty threshold equal to 0.6 times the mean
svyfgt(~eqincome, des_eusilc_rep, g=1, type_thresh= "relm", thresh = TRUE)

\dontrun{

# database-backed design
library(RSQLite)
library(DBI)
dbfile <- tempfile()
conn <- dbConnect( RSQLite::SQLite() , dbfile )
dbWriteTable( conn , 'eusilc' , eusilc )

dbd_eusilc <-
	svydesign(
		ids = ~rb030 ,
		strata = ~db040 ,
		weights = ~rb050 ,
		data="eusilc",
		dbname=dbfile,
		dbtype="SQLite"
	)


dbd_eusilc <- convey_prep( dbd_eusilc )

# headcount ratio, poverty threshold fixed
svyfgt(~eqincome, dbd_eusilc, g=0, abs_thresh=10000)
# poverty gap index, poverty threshold fixed
svyfgt(~eqincome, dbd_eusilc, g=1, abs_thresh=10000)
# headcount ratio, poverty threshold equal to arpt
svyfgt(~eqincome, dbd_eusilc, g=0, type_thresh= "relq", thresh = TRUE)
# poverty gap index, poverty threshold equal to arpt
svyfgt(~eqincome, dbd_eusilc, g=1, type_thresh= "relq")
# headcount ratio, poverty threshold equal to .6 times the mean
svyfgt(~eqincome, dbd_eusilc, g=0, type_thresh= "relm")
# poverty gap index, poverty threshold equal to 0.6 times the mean
svyfgt(~eqincome, dbd_eusilc, g=1, type_thresh= "relm")

dbRemoveTable( conn , 'eusilc' )

dbDisconnect( conn , shutdown = TRUE )

}

}
\references{
James Foster, Joel Greer and Erik Thorbecke (1984). A class of decomposable poverty measures.
\emph{Econometrica}, Vol.52, No.3, pp. 761-766.

Y.G. Berger and C. J. Skinner (2003), Variance estimation for a low income proportion.
\emph{Journal of the Royal Statistical Society: Series C (Applied Statistics)}, Vol. 52, No. 4, pp. 457-468.
DOI \doi{10.1111/1467-9876.00417}

Buhong Zheng (2001). Statistical inference for poverty measures with relative poverty lines.
\emph{Journal of Econometrics}, Vol. 101, pp. 337-356.

Guillaume Osier (2009). Variance estimation for complex indicators
of poverty and inequality. \emph{Journal of the European Survey Research
Association}, Vol.3, No.3, pp. 167-195,
ISSN 1864-3361, URL \url{https://ojs.ub.uni-konstanz.de/srm/article/view/369}.

Jean-Claude Deville (1999). Variance estimation for complex statistics and estimators:
linearization and residual techniques. Survey Methodology, 25, 193-203,
URL \url{https://www150.statcan.gc.ca/n1/en/catalogue/12-001-X19990024882}.
}
\seealso{
\code{\link{svyarpt}}
}
\author{
Djalma Pessoa, Anthony Damico, and Guilherme Jacob
}
\keyword{survey}
