% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posixt.R
\name{date-time-parse}
\alias{date-time-parse}
\alias{date_time_parse}
\alias{date_time_parse_complete}
\alias{date_time_parse_abbrev}
\title{Parsing: date-time}
\usage{
date_time_parse(
  x,
  zone,
  ...,
  format = NULL,
  locale = clock_locale(),
  nonexistent = NULL,
  ambiguous = NULL
)

date_time_parse_complete(x, ..., format = NULL, locale = clock_locale())

date_time_parse_abbrev(x, zone, ..., format = NULL, locale = clock_locale())
}
\arguments{
\item{x}{\verb{[character]}

A character vector to parse.}

\item{zone}{\verb{[character(1)]}

A full time zone name.}

\item{...}{These dots are for future extensions and must be empty.}

\item{format}{\verb{[character / NULL]}

A format string. A combination of the following commands, or \code{NULL},
in which case a default format string is used.

A vector of multiple format strings can be supplied. They will be tried in
the order they are provided.

\strong{Year}
\itemize{
\item \verb{\%C}: The century as a decimal number. The modified command \verb{\%NC} where
\code{N} is a positive decimal integer specifies the maximum number of
characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%y}: The last two decimal digits of the year. If the century is not
otherwise specified (e.g. with \verb{\%C}), values in the range \verb{[69 - 99]} are
presumed to refer to the years \verb{[1969 - 1999]}, and values in the range
\verb{[00 - 68]} are presumed to refer to the years \verb{[2000 - 2068]}. The
modified command \verb{\%Ny}, where \code{N} is a positive decimal integer, specifies
the maximum number of characters to read. If not specified, the default is
\code{2}. Leading zeroes are permitted but not required.
\item \verb{\%Y}: The year as a decimal number. The modified command \verb{\%NY} where \code{N}
is a positive decimal integer specifies the maximum number of characters to
read. If not specified, the default is \code{4}. Leading zeroes are permitted
but not required.
}

\strong{Month}
\itemize{
\item \verb{\%b}, \verb{\%B}, \verb{\%h}: The \code{locale}'s full or abbreviated case-insensitive
month name.
\item \verb{\%m}: The month as a decimal number. January is \code{1}. The modified command
\verb{\%Nm} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
}

\strong{Day}
\itemize{
\item \verb{\%d}, \verb{\%e}: The day of the month as a decimal number. The modified
command \verb{\%Nd} where \code{N} is a positive decimal integer specifies the maximum
number of characters to read. If not specified, the default is \code{2}. Leading
zeroes are permitted but not required.
}

\strong{Day of the week}
\itemize{
\item \verb{\%a}, \verb{\%A}: The \code{locale}'s full or abbreviated case-insensitive weekday
name.
\item \verb{\%w}: The weekday as a decimal number (\code{0-6}), where Sunday is \code{0}. The
modified command \verb{\%Nw} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{1}. Leading zeroes are permitted but not required.
}

\strong{ISO 8601 week-based year}
\itemize{
\item \verb{\%g}: The last two decimal digits of the ISO week-based year. The
modified command \verb{\%Ng} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{2}. Leading zeroes are permitted but not required.
\item \verb{\%G}: The ISO week-based year as a decimal number. The modified command
\verb{\%NG} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{4}. Leading zeroes
are permitted but not required.
\item \verb{\%V}: The ISO week-based week number as a decimal number. The modified
command \verb{\%NV} where \code{N} is a positive decimal integer specifies the maximum
number of characters to read. If not specified, the default is \code{2}. Leading
zeroes are permitted but not required.
\item \verb{\%u}: The ISO weekday as a decimal number (\code{1-7}), where Monday is \code{1}.
The modified command \verb{\%Nu} where \code{N} is a positive decimal integer
specifies the maximum number of characters to read. If not specified, the
default is \code{1}. Leading zeroes are permitted but not required.
}

\strong{Week of the year}
\itemize{
\item \verb{\%U}: The week number of the year as a decimal number. The first Sunday
of the year is the first day of week \code{01}. Days of the same year prior to
that are in week \code{00}. The modified command \verb{\%NU} where \code{N} is a positive
decimal integer specifies the maximum number of characters to read. If not
specified, the default is \code{2}. Leading zeroes are permitted but not
required.
\item \verb{\%W}: The week number of the year as a decimal number. The first Monday
of the year is the first day of week \code{01}. Days of the same year prior to
that are in week \code{00}. The modified command \verb{\%NW} where \code{N} is a positive
decimal integer specifies the maximum number of characters to read. If not
specified, the default is \code{2}. Leading zeroes are permitted but not
required.
}

\strong{Day of the year}
\itemize{
\item \verb{\%j}: The day of the year as a decimal number. January 1 is \code{1}. The
modified command \verb{\%Nj} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{3}. Leading zeroes are permitted but not required.
}

\strong{Date}
\itemize{
\item \verb{\%D}, \verb{\%x}: Equivalent to \verb{\%m/\%d/\%y}.
\item \verb{\%F}: Equivalent to \verb{\%Y-\%m-\%d}. If modified with a width (like \verb{\%NF}),
the width is applied to only \verb{\%Y}.
}

\strong{Time of day}
\itemize{
\item \verb{\%H}: The hour (24-hour clock) as a decimal number. The modified command
\verb{\%NH} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%I}: The hour (12-hour clock) as a decimal number. The modified command
\verb{\%NI} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%M}: The minutes as a decimal number. The modified command \verb{\%NM} where
\code{N} is a positive decimal integer specifies the maximum number of
characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%S}: The seconds as a decimal number. The modified command \verb{\%NS} where
\code{N} is a positive decimal integer specifies the maximum number of
characters to read. If not specified, the default is determined by the
precision that you are parsing at. If encountered, the \code{locale}
determines the decimal point character. Leading zeroes are permitted but
not required.
\item \verb{\%p}: The \code{locale}'s equivalent of the AM/PM designations associated with
a 12-hour clock. The command \verb{\%I} must precede \verb{\%p} in the format string.
\item \verb{\%R}: Equivalent to \verb{\%H:\%M}.
\item \verb{\%T}, \verb{\%X}: Equivalent to \verb{\%H:\%M:\%S}.
\item \verb{\%r}: Equivalent to \verb{\%I:\%M:\%S \%p}.
}

\strong{Time zone}
\itemize{
\item \verb{\%z}: The offset from UTC in the format \verb{[+|-]hh[mm]}. For example
\code{-0430} refers to 4 hours 30 minutes behind UTC. And \code{04} refers to 4 hours
ahead of UTC. The modified command \verb{\%Ez} parses a \code{:} between the hours and
minutes and leading zeroes on the hour field are optional:
\verb{[+|-]h[h][:mm]}. For example \code{-04:30} refers to 4 hours 30 minutes behind
UTC. And \code{4} refers to 4 hours ahead of UTC.
\item \verb{\%Z}: The full time zone name or the time zone abbreviation, depending on
the function being used. A single word is parsed. This word can only
contain characters that are alphanumeric, or one of \code{'_'}, \code{'/'}, \code{'-'} or
\code{'+'}.
}

\strong{Miscellaneous}
\itemize{
\item \verb{\%c}: A date and time representation. Equivalent to
\verb{\%a \%b \%d \%H:\%M:\%S \%Y}.
\item \code{\%\%}: A \verb{\%} character.
\item \verb{\%n}: Matches one white space character. \verb{\%n}, \verb{\%t}, and a space can be
combined to match a wide range of white-space patterns. For example \code{"\%n "}
matches one or more white space characters, and \code{"\%n\%t\%t"} matches one to
three white space characters.
\item \verb{\%t}: Matches zero or one white space characters.
}}

\item{locale}{\verb{[clock_locale]}

A locale object created from \code{\link[=clock_locale]{clock_locale()}}.}

\item{nonexistent}{\verb{[character / NULL]}

One of the following nonexistent time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"roll-forward"}: The next valid instant in time.
\item \code{"roll-backward"}: The previous valid instant in time.
\item \code{"shift-forward"}: Shift the nonexistent time forward by the size of
the daylight saving time gap.
\item \verb{"shift-backward}: Shift the nonexistent time backward by the size of
the daylight saving time gap.
\item \code{"NA"}: Replace nonexistent times with \code{NA}.
\item \code{"error"}: Error on nonexistent times.
}

Using either \code{"roll-forward"} or \code{"roll-backward"} is generally
recommended over shifting, as these two strategies maintain the
\emph{relative ordering} between elements of the input.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{nonexistent} must be supplied
and cannot be \code{NULL}. This is a convenient way to make production code
robust to nonexistent times.}

\item{ambiguous}{\verb{[character / zoned_time / POSIXct / list(2) / NULL]}

One of the following ambiguous time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"earliest"}: Of the two possible times, choose the earliest one.
\item \code{"latest"}: Of the two possible times, choose the latest one.
\item \code{"NA"}: Replace ambiguous times with \code{NA}.
\item \code{"error"}: Error on ambiguous times.
}

Alternatively, \code{ambiguous} is allowed to be a zoned_time (or POSIXct) that
is either length 1, or the same length as the input. If an ambiguous time
is encountered, the zoned_time is consulted. If the zoned_time corresponds
to a naive_time that is also ambiguous \emph{and} uses the same daylight saving
time transition point as the original ambiguous time, then the offset of
the zoned_time is used to resolve the ambiguity. If the ambiguity cannot be
resolved by consulting the zoned_time, then this method falls back to
\code{NULL}.

Finally, \code{ambiguous} is allowed to be a list of size 2, where the first
element of the list is a zoned_time (as described above), and the second
element of the list is an ambiguous time resolution strategy to use when
the ambiguous time cannot be resolved by consulting the zoned_time.
Specifying a zoned_time on its own is identical to \verb{list(<zoned_time>, NULL)}.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{ambiguous} must be supplied and
cannot be \code{NULL}. Additionally, \code{ambiguous} cannot be specified as a
zoned_time on its own, as this implies \code{NULL} for ambiguous times that the
zoned_time cannot resolve. Instead, it must be specified as a list
alongside an ambiguous time resolution strategy as described above. This is
a convenient way to make production code robust to ambiguous times.}
}
\value{
A POSIXct.
}
\description{
There are three parsers for parsing strings into POSIXct date-times,
\code{date_time_parse()}, \code{date_time_parse_complete()}, and
\code{date_time_parse_abbrev()}.
\subsection{date_time_parse()}{

\code{date_time_parse()} is useful for strings like \code{"2019-01-01 00:00:00"}, where
the UTC offset and full time zone name are not present in the string. The
string is first parsed as a naive-time without any time zone assumptions, and
is then converted to a POSIXct with the supplied \code{zone}.

Because converting from naive-time to POSIXct may result in nonexistent or
ambiguous times due to daylight saving time, these must be resolved
explicitly with the \code{nonexistent} and \code{ambiguous} arguments.

\code{date_time_parse()} completely ignores the \verb{\%z} and \verb{\%Z} commands. The only
time zone specific information that is used is the \code{zone}.

The default \code{format} used is \code{"\%Y-\%m-\%d \%H:\%M:\%S"}.
}

\subsection{date_time_parse_complete()}{

\code{date_time_parse_complete()} is a parser for \emph{complete} date-time strings,
like \code{"2019-01-01 00:00:00-05:00[America/New_York]"}. A complete date-time
string has both the time zone offset and full time zone name in the string,
which is the only way for the string itself to contain all of the information
required to construct a zoned-time. Because of this,
\code{date_time_parse_complete()} requires both the \verb{\%z} and \verb{\%Z} commands to be
supplied in the \code{format} string.

The default \code{format} used is \code{"\%Y-\%m-\%d \%H:\%M:\%S\%Ez[\%Z]"}.
}

\subsection{date_time_parse_abbrev()}{

\code{date_time_parse_abbrev()} is a parser for date-time strings containing only
a time zone abbreviation, like \code{"2019-01-01 00:00:00 EST"}. The time zone
abbreviation is not enough to identify the full time zone name that the
date-time belongs to, so the full time zone name must be supplied as the
\code{zone} argument. However, the time zone abbreviation can help with resolving
ambiguity around daylight saving time fallbacks.

For \code{date_time_parse_abbrev()}, \verb{\%Z} must be supplied and is interpreted as
the time zone abbreviation rather than the full time zone name.

If used, the \verb{\%z} command must parse correctly, but its value will be
completely ignored.

The default \code{format} used is \code{"\%Y-\%m-\%d \%H:\%M:\%S \%Z"}.
}
}
\details{
If \code{date_time_parse_complete()} is given input that is length zero, all
\code{NA}s, or completely fails to parse, then no time zone will be able to be
determined. In that case, the result will use \code{"UTC"}.

If manually parsing sub-second components, be aware that they will be
automatically rounded to the nearest second when converting them to POSIXct.
See the examples for a way to control this.
}
\examples{
# Parse with a known `zone`, even though that information isn't in the string
date_time_parse("2020-01-01 05:06:07", "America/New_York")

# Same time as above, except this is a completely unambiguous parse that
# doesn't require a `zone` argument, because the zone name and offset are
# both present in the string
date_time_parse_complete("2020-01-01 05:06:07-05:00[America/New_York]")

# Only day components
date_time_parse("2020-01-01", "America/New_York", format = "\%Y-\%m-\%d")

# `date_time_parse()` may have issues with ambiguous times due to daylight
# saving time fallbacks. For example, there were two 1'oclock hours here:
x <- date_time_parse("1970-10-25 00:59:59", "America/New_York")

# First (earliest) 1'oclock hour
add_seconds(x, 1)
# Second (latest) 1'oclock hour
add_seconds(x, 3601)

# If you try to parse this ambiguous time directly, you'll get an error:
ambiguous_time <- "1970-10-25 01:00:00"
try(date_time_parse(ambiguous_time, "America/New_York"))

# Resolve it by specifying whether you'd like to use the
# `earliest` or `latest` of the two possible times
date_time_parse(ambiguous_time, "America/New_York", ambiguous = "earliest")
date_time_parse(ambiguous_time, "America/New_York", ambiguous = "latest")

# `date_time_parse_complete()` doesn't have these issues, as it requires
# that the offset and zone name are both in the string, which resolves
# the ambiguity
complete_times <- c(
  "1970-10-25 01:00:00-04:00[America/New_York]",
  "1970-10-25 01:00:00-05:00[America/New_York]"
)
date_time_parse_complete(complete_times)

# `date_time_parse_abbrev()` also doesn't have these issues, since it
# uses the time zone abbreviation name to resolve the ambiguity
abbrev_times <- c(
  "1970-10-25 01:00:00 EDT",
  "1970-10-25 01:00:00 EST"
)
date_time_parse_abbrev(abbrev_times, "America/New_York")

# ---------------------------------------------------------------------------
# Rounding of sub-second components

# Generally, if you have a string with sub-second components, they will
# be ignored when parsing into a date-time
x <- c("2019-01-01 00:00:01.1", "2019-01-01 00:00:01.7")

date_time_parse(x, "America/New_York")

# If you manually try and parse those sub-second components with `\%4S` to
# read the 2 seconds, 1 decimal point, and 1 fractional component, the
# fractional component will be rounded to the nearest second
date_time_parse(x, "America/New_York", format = "\%Y-\%m-\%d \%H:\%M:\%4S")

# If you don't like this, parse the full string as a naive-time,
# then round manually and convert to a POSIXct
nt <- naive_time_parse(x, format = "\%Y-\%m-\%d \%H:\%M:\%S", precision = "millisecond")
nt

nt <- time_point_floor(nt, "second")
nt

as.POSIXct(nt, "America/New_York")
}
