#' Visualise climate cross correlation or autocorrelation.
#' 
#' Create a colour plot to visualise the results of \code{\link{autowin}} or
#' \code{\link{crosswin}}. Displays correlation across all desired climate
#' windows.
#' @param cor.output Output of \code{\link{autowin}} or
#'   \code{\link{crosswin}}
#' @param type Should be either "A" for data generated by \code{\link{autowin}}
#'   or "C" for data generated by \code{\link{crosswin}}.
#' @return Will generate a colour plot to visualise the correlation data.
#' @author Liam D. Bailey and Martijn van de Pol
#' @import ggplot2
#' @examples
#' \dontrun{
#'# Visualise climate autocorrelation
#'
#'data(Mass)
#'data(MassClimate)
#' 
#'# Fit a single climate window using the datasets Mass and MassClimate.
#' 
#'single <- singlewin(xvar = list(Temp = MassClimate$Temp), 
#'                    cdate = MassClimate$Date, bdate = Mass$Date,
#'                    baseline = lm(Mass ~ 1, data = Mass), 
#'                    furthest = 72, closest = 15,
#'                    stat = "mean", func = "lin",
#'                    type = "fixed", cutoff.day = 20, cutoff.month = 5,
#'                    cmissing = FALSE, cinterval = "day")            
#' 
#'# Test the autocorrelation between the climate in this single window and other climate windows.
#' 
#'auto <- autowin(reference = single$BestModelData$climate,
#'                xvar  = list(Temp = MassClimate$Temp), 
#'                cdate = MassClimate$Date, bdate = Mass$Date,
#'                baseline = lm(Mass ~ 1, data = Mass), 
#'                furthest = 365, closest = 0, 
#'                stat = "mean", func = "lin",
#'                type = "fixed", cutoff.day = 20, cutoff.month = 5,
#'                cmissing = FALSE, cinterval = "day")
#'                 
#'# Plot the auto-correlation data
#' 
#'plotcor(auto, type = "A")
#'}
#'@export


plotcor <- function(cor.output, type = type){
  ifelse (type == "A", title <- "Correlation between single window and all other windows", 
          title <- "Correlation between 2 climate variables in all windows")
  with(cor.output, {
    ggplot(cor.output, aes(x = WindowClose, y = WindowOpen, z = cor))+
      geom_tile(aes(fill = cor))+
      scale_fill_gradient2(low = "red", mid = "yellow", high = "blue", 
                           midpoint = mean(cor.output$cor), name = "")+
      theme_classic()+
      theme(panel.grid.major = element_blank(),
            panel.grid.minor = element_blank(),
            axis.line = element_line(size = 0.25, colour = "black"))+
      ggtitle(title)+
      ylab("Window open")+
      xlab("Window close")
  }
  )
}