% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dnn.R
\name{dnn}
\alias{dnn}
\title{DNN}
\usage{
dnn(
  formula,
  data = NULL,
  loss = c("mse", "mae", "softmax", "cross-entropy", "gaussian", "binomial", "poisson"),
  hidden = c(10L, 10L, 10L),
  activation = c("relu", "leaky_relu", "tanh", "elu", "rrelu", "prelu", "softplus",
    "celu", "selu", "gelu", "relu6", "sigmoid", "softsign", "hardtanh", "tanhshrink",
    "softshrink", "hardshrink", "log_sigmoid"),
  validation = 0,
  bias = TRUE,
  lambda = 0,
  alpha = 0.5,
  dropout = 0,
  optimizer = c("sgd", "adam", "adadelta", "adagrad", "rmsprop", "rprop"),
  lr = 0.01,
  batchsize = 32L,
  shuffle = TRUE,
  epochs = 100,
  plot = TRUE,
  verbose = TRUE,
  lr_scheduler = NULL,
  custom_parameters = NULL,
  device = c("cpu", "cuda"),
  early_stopping = FALSE
)
}
\arguments{
\item{formula}{an object of class "\code{\link[stats]{formula}}": a description of the model that should be fitted}

\item{data}{matrix or data.frame}

\item{loss}{loss after which network should be optimized. Can also be distribution from the stats package or own function}

\item{hidden}{hidden units in layers, length of hidden corresponds to number of layers}

\item{activation}{activation functions, can be of length one, or a vector of different activation functions for each layer}

\item{validation}{percentage of data set that should be taken as validation set (chosen randomly)}

\item{bias}{whether use biases in the layers, can be of length one, or a vector (number of hidden layers + 1 (last layer)) of logicals for each layer.}

\item{lambda}{strength of regularization: lambda penalty, \eqn{\lambda * (L1 + L2)} (see alpha)}

\item{alpha}{add L1/L2 regularization to training  \eqn{(1 - \alpha) * |weights| + \alpha ||weights||^2} will get added for each layer. Can be single integer between 0 and 1 or vector of alpha values if layers should be regularized differently.}

\item{dropout}{dropout rate, probability of a node getting left out during training (see \code{\link[torch]{nn_dropout}})}

\item{optimizer}{which optimizer used for training the network, for more adjustments to optimizer see \code{\link{config_optimizer}}}

\item{lr}{learning rate given to optimizer}

\item{batchsize}{number of samples that are used to calculate one learning rate step}

\item{shuffle}{if TRUE, data in each batch gets reshuffled every epoch}

\item{epochs}{epochs the training goes on for}

\item{plot}{plot training loss}

\item{verbose}{print training and validation loss of epochs}

\item{lr_scheduler}{learning rate scheduler created with \code{\link{config_lr_scheduler}}}

\item{custom_parameters}{List of parameters/variables to be optimized. Can be used in a custom loss function. See Vignette for example.}

\item{device}{device on which network should be trained on.}

\item{early_stopping}{if set to integer, training will stop if loss has gotten higher for defined number of epochs in a row, will use validation loss is available.}
}
\value{
an S3 object of class \code{"cito.dnn"} is returned. It is a list containing everything there is to know about the model and its training process.
The list consists of the following attributes:
\item{net}{An object of class "nn_sequential" "nn_module", originates from the torch package and represents the core object of this workflow.}
\item{call}{The original function call}
\item{loss}{A list which contains relevant information for the target variable and the used loss function}
\item{data}{Contains data used for training the model}
\item{weigths}{List of weights for each training epoch}
\item{use_model_epoch}{Integer, which defines which model from which training epoch should be used for prediction.}
\item{loaded_model_epoch}{Integer, shows which model from which epoch is loaded currently into model$net.}
\item{model_properties}{A list of properties of the neural network, contains number of input nodes, number of output nodes, size of hidden layers, activation functions, whether bias is included and if dropout layers are included.}
\item{training_properties}{A list of all training parameters that were used the last time the model was trained. It consists of learning rate, information about an learning rate scheduler, information about the optimizer, number of epochs, whether early stopping was used, if plot was active, lambda and alpha for L1/L2 regularization, batchsize, shuffle, was the data set split into validation and training, which formula was used for training and at which epoch did the training stop.}
\item{losses}{A data.frame containing training and validation losses of each epoch}
}
\description{
fits a custom deep neural network. dnn() supports the formula syntax and allows to customize the neural network to a maximal degree.
So far, only Multilayer Perceptrons are possible.
}
\details{
In a Multilayer Perceptron (MLP) network every neuron is connected with all neurons of the previous layer and connected to all neurons of the layer afterwards.
The value of each neuron is calculated with:

\eqn{ a (\sum_j{ w_j * a_j})}

Where \eqn{w_j} is the weight and \eqn{a_j} is the value from neuron j to the current one. a() is the activation function, e.g. \eqn{ relu(x) = max(0,x)}
As regularization methods there is dropout and elastic net regularization available. These methods help you avoid over fitting.

Training on graphic cards:
If you want to train on your cuda devide, you have to install the NVIDIA CUDA toolkit version 11.3. and cuDNN 8.4. beforehand. Make sure that you have exactly these versions installed, since it does not work with other version.
For more information see \href{https://torch.mlverse.org/docs/articles/installation.html}{mlverse: 'torch'}
}
\examples{
\donttest{
if(torch::torch_is_installed()){
library(cito)

set.seed(222)
validation_set<- sample(c(1:nrow(datasets::iris)),25)

# Build and train  Network
nn.fit<- dnn(Sepal.Length~., data = datasets::iris[-validation_set,])

# Sturcture of Neural Network
print(nn.fit)

# Use model on validation set
predictions <- predict(nn.fit, iris[validation_set,])

# Scatterplot
plot(iris[validation_set,]$Sepal.Length,predictions)
# MAE
mean(abs(predictions-iris[validation_set,]$Sepal.Length))

# Get variable importances
summary(nn.fit)

# Partial dependencies
PDP(nn.fit, variable = "Petal.Length")

# Accumulated local effect plots
ALE(nn.fit, variable = "Petal.Length")

# Custom loss functions and additional parameters
## Normal Likelihood with sd parameter:
custom_loss = function(true, pred) {
  logLik = torch::distr_normal(pred,
                               scale = torch::nnf_relu(scale)+
                                 0.001)$log_prob(true)
  return(-logLik$mean())
}

nn.fit<- dnn(Sepal.Length~.,
             data = datasets::iris[-validation_set,],
             loss = custom_loss,
             custom_parameters = list(scale = 1.0)
)
nn.fit$parameter$scale

## Multivariate normal likelihood with parametrized covariance matrix
## Sigma = L*L^t + D
## Helper function to build covariance matrix
create_cov = function(LU, Diag) {
  return(torch::torch_matmul(LU, LU$t()) + torch::torch_diag(Diag+0.01))
}

custom_loss_MVN = function(true, pred) {
  Sigma = create_cov(SigmaPar, SigmaDiag)
  logLik = torch::distr_multivariate_normal(pred,
                                            covariance_matrix = Sigma)$
    log_prob(true)
  return(-logLik$mean())
}


nn.fit<- dnn(cbind(Sepal.Length, Sepal.Width, Petal.Length)~.,
             data = datasets::iris[-validation_set,],
             lr = 0.01,
             loss = custom_loss_MVN,
             custom_parameters =
               list(SigmaDiag =  rep(1, 3),
                    SigmaPar = matrix(rnorm(6, sd = 0.001), 3, 2))
)
as.matrix(create_cov(nn.fit$loss$parameter$SigmaPar,
                     nn.fit$loss$parameter$SigmaDiag))

}
}
}
\seealso{
\code{\link{predict.citodnn}}, \code{\link{plot.citodnn}},  \code{\link{coef.citodnn}},\code{\link{print.citodnn}}, \code{\link{summary.citodnn}}, \code{\link{continue_training}}, \code{\link{analyze_training}}, \code{\link{PDP}}, \code{\link{ALE}},
}
