% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/checkpoint.R
\name{checkpoint}
\alias{checkpoint}
\title{Configures R session to use packages as they existed on CRAN at time of snapshot.}
\usage{
checkpoint(snapshotDate, project = getwd(), R.version,
  scanForPackages = TRUE, checkpointLocation = "~/", verbose = TRUE,
  use.knitr, auto.install.knitr = TRUE, scan.rnw.with.knitr = FALSE,
  forceInstall = FALSE, forceProject = FALSE, use.lock = TRUE)
}
\arguments{
\item{snapshotDate}{Date of snapshot to use in \code{YYYY-MM-DD} format, e.g. \code{"2014-09-17"}.  Specify a date on or after \code{"2014-09-17"}.  MRAN takes one snapshot per day. To list all valid snapshot dates on MRAN use \code{\link[=getValidSnapshots]{getValidSnapshots()}}}

\item{project}{A project path.  This is the path to the root of the project that references the packages to be installed from the MRAN snapshot for the date specified for \code{snapshotDate}.  Defaults to current working directory using \code{\link[=getwd]{getwd()}}.}

\item{R.version}{Optional character string, e.g. \code{"3.1.2"}.  If specified, compares the current \link{R.version} to the specified R.version. If these differ, stops processing with an error, making no changes to the system. Specifically, if the check fails, the library path is NOT modified. This argument allows the original script author to specify a specific version of R to obtain the desired results.}

\item{scanForPackages}{If \code{TRUE}, scans for packages in project folder (see details). If FALSE, skips the scanning process.  A use case for \code{scanForPackages = FALSE} is to skip the scanning and installation process, e.g. in production environments with a large number of R scripts in the project.  Only set \code{scanForPackages = FALSE} if you are certain that all package dependencies are already in the checkpoint folder.}

\item{checkpointLocation}{File path where the checkpoint library is stored.  Default is \code{"~/"}, i.e. the user's home directory. A use case for changing this is to create a checkpoint library on a portable drive (e.g. USB drive).}

\item{verbose}{If \code{TRUE}, displays progress messages.}

\item{use.knitr}{If \code{TRUE},  parses all \code{Rmarkdown} files using the \code{knitr} package.}

\item{auto.install.knitr}{If \code{TRUE} and the project contains rmarkdown files, then automatically included the packages \code{knitr} in packages to install.}

\item{scan.rnw.with.knitr}{If \code{TRUE}, uses \code{\link[knitr:knit]{knitr::knit()}} to parse \code{.Rnw} files, otherwise use \code{\link[utils:Sweave]{utils::Sweave()}}}

\item{forceInstall}{If \code{TRUE}, forces the re-installation of all discovered packages and their dependencies. This is useful if, for some reason, the checkpoint archive becomes corrupted.}

\item{forceProject}{If \code{TRUE}, forces the checkpoint process, even if the provided project folder doesn't look like an R project. A commonly reported user problem is that they accidentally trigger the checkpoint process from their home folder, resulting in scanning many R files and downloading many packages. To prevent this, we use a heuristic to determine if the project folder looks like an R project. If the project folder is the home folder, and also contains no R files, then \code{checkpoint()} asks for confirmation to continue.}

\item{use.lock}{if \code{FALSE}, sets the \code{--no-lock} argument to \code{R CMD INSTALL}.}
}
\value{
Checkpoint is called for its side-effects (see the details section), but invisibly returns a list with elements:
\itemize{
\item \code{files_not_scanned}
\item \code{pkgs_found}
\item \code{pkgs_not_on_mran}
\item \code{pkgs_installed}
}
}
\description{
Together, the checkpoint package and the checkpoint server act as a CRAN time machine.  The \code{checkpoint()} function installs the packages referenced in the specified project to a local library exactly as they existed at the specified point in time.  Only those packages are available to your session, thereby avoiding any package updates that came later and may have altered your results.  In this way, anyone using the checkpoint \code{checkpoint()} function can ensure the reproducibility of your scripts or projects at any time.
}
\section{Details}{


\code{checkpoint()} creates a local library into which it installs a copy of the packages required by your project as they existed on CRAN on the specified snapshot date.  Your R session is updated to use only these packages.

To automatically determine all packages used in your project, the function scans all R code (\code{.R}, \code{.Rmd}, and \code{.Rpres} files) for \code{\link[=library]{library()}} and \code{\link[=require]{require()}} statements. In addition, scans for occurrences of code that accesses functions in namespaces using \code{package[::]foo()} and \code{package[:::]foo()}. Finally, any occurrences of the functions \link[methods:setClass]{methods::setClass}, \link[methods:setRefClass]{methods::setRefClass}, \link[methods:setMethod]{methods::setMethod} or \link[methods:setGeneric]{methods::setGeneric} will also identify the \code{methods} package as a dependency.

Specifically, the function will:
\itemize{
\item Create a new local snapshot library to install packages.  By default this library folder is at \code{~/.checkpoint} but you can modify the path using the \code{checkpointLocation} argument.
\item Update the options for your CRAN mirror and point to an MRAN snapshot using \link{options}\code{(repos)}
\item Scan your project folder for all required packages and install them from the snapshot using \code{\link[utils:install.packages]{utils::install.packages()}}
}
}

\section{Resetting the checkpoint}{


To reset the checkpoint, simply restart your R session.

You can also use the experimental function \code{\link[=unCheckpoint]{unCheckpoint()}}
}

\section{Changing the default MRAN url}{


By default, \code{checkpoint()} uses https to download packages. The default MRAN snapshot defaults to \url{https://mran.microsoft.com/snapshot} in R versions 3.2.0 and later, if https support is enabled.

You can modify the default URL. To change the URL, use \code{options(checkpoint.mranUrl = ...)}.
}

\section{Log file}{


As a side effect, the \code{checkpoint} function writes a log file with information about the downloaded files, in particular the package downloaded and the associated file size in bytes. The log is stored at the root of the \code{checkpointLocation}. For example, if \code{checkpointLocation} is the user home folder (the default) then the log file is at \code{~/.checkpoint/checkpoint_log.csv}. This file contains columns for:
\itemize{
\item \code{timestamp}
\item \code{snapshotDate}
\item \code{pkg}
\item \code{bytes}
}
}

\section{Last accessed date}{


The \code{\link[=checkpoint]{checkpoint()}} function stores a marker in the snapshot folder every time the function gets called. This marker contains the system date, thus indicating the the last time the snapshot was accessed.  See also \code{\link[=getAccessDate]{getAccessDate()}}. To remove snapshots that have not been used since a given date, use \code{\link[=checkpointRemove]{checkpointRemove()}}
}

\examples{
\dontrun{

# Create temporary project and set working directory

example_project <- paste0("~/checkpoint_example_project_", Sys.Date())

dir.create(example_project, recursive = TRUE)
oldwd <- setwd(example_project)


# Write dummy code file to project

cat("library(MASS)", "library(foreach)",
    sep="\\n",
    file="checkpoint_example_code.R")


# Create a checkpoint by specifying a snapshot date

library(checkpoint)
checkpoint("2014-09-17")

# Check that CRAN mirror is set to MRAN snapshot
getOption("repos")

# Check that library path is set to ~/.checkpoint
.libPaths()

# Check which packages are installed in checkpoint library
installed.packages()

# cleanup
unlink(example_project, recursive = TRUE)
setwd(oldwd)

}

}
\seealso{
Other checkpoint functions: \code{\link{checkpointArchives}},
  \code{\link{checkpointRemove}},
  \code{\link{getAccessDate}},
  \code{\link{getValidSnapshots}}, \code{\link{mranUrl}},
  \code{\link{setSnapshot}}, \code{\link{unCheckpoint}}
}
\concept{checkpoint functions}
