% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pocr.R
\name{pocr}
\alias{pocr}
\title{Product-of-Coefficients-Regression Estimation Method}
\usage{
pocr(fit.x = NULL, fit.m, fit.y, treat, covariates, sims = 100, conf.level = .95,
     cluster = NULL, long = TRUE, mc.cores = 1L, seed = NULL)
}
\arguments{
\item{fit.x}{a fitted model object for intermediate confounder. Can be of class 'lm'.
Only necessary if the mediator is categorical. Default is 'NULL'.}

\item{fit.m}{a fitted model object for mediator. Can be of class 'lm', 'glm',
'multinom', or 'polr'.}

\item{fit.y}{a fitted model object for outcome. Can be of class 'lm'.}

\item{treat}{a character string indicating the name of the treatment variable
used in the models. The treatment can be categorical with two or more
categories (two- or multi-valued factor).}

\item{covariates}{a vector containing the name of the covariate variable(s)
used in the models. Each covariate can be categorical with two or more
categories (two- or multi-valued factor) or continuous (numeric).}

\item{sims}{number of Monte Carlo draws for nonparametric bootstrap.}

\item{conf.level}{level of the returned two-sided confidence intervals,
which are estimated by the nonparametric percentile bootstrap method.
Default is to return the 2.5 and 97.5 percentiles of the simulated quantities.}

\item{cluster}{a vector of cluster indicators for the bootstrap. If provided,
the cluster bootstrap is used. Default is 'NULL'.}

\item{long}{a logical value. If 'TRUE', the output will contain the entire
sets of estimates for all bootstrap samples. Default is 'TRUE'.}

\item{mc.cores}{The number of cores to use. Must be exactly 1 on Windows.}

\item{seed}{seed number for the reproducibility of results. Default is `NULL'.}
}
\value{
\item{result}{a matrix containing the point estimates of the initial disparity,
disparity remaining, and disparity reduction, and the percentile bootstrap
confidence intervals for each estimate.}
\item{all.result}{a matrix containing the point estimates of the initial disparity,
disparity remaining, and disparity reduction for all bootstrap samples. Returned
if 'long' is 'TRUE'.}
}
\description{
'pocr' is used to estimate the initial disparity, disparity reduction, and
disparity remaining for causal decomposition analysis, using the
product-of-coefficients-regression estimation method proposed by Park et al. (2021+).
}
\details{
This function returns the point estimates of the initial disparity,
disparity reduction, and disparity remaining for a categorical
treatment and a variety of types of outcome and mediator(s) in causal
decomposition analysis. It also returns nonparametric
percentile bootstrap confidence intervals for each estimate.

The definition of the initial disparity, disparity reduction, and
disparity remaining can be found in help('mmi'). As opposed to the 'mmi' and
'smi' function, this function uses the product-of-coefficients-regression method
suggested by Park et al. (2021+). It always make the inference
conditional on baseline covariates. Therefore, users need to center the data
before fitting the models. See the reference for more details.

As of version 0.1.0, the mediator model ('fit.m') can be of class 'lm', 'glm',
'multinom', or 'polr', corresponding respectively to the linear regression
models and generalized linear models, multinomial log-linear models, and
ordered response models. The outcome model ('fit.y') can be of class 'lm'.
The intermediate confounder model ('fit.x') can also be of class 'lm' and only
necessary when the mediator is categorical.
}
\examples{
data(sdata)

# To be conditional on covariates, first create data with centered covariates
# copy data
sdata.c <- sdata
# center quantitative covariate(s)
sdata.c$C.num <- scale(sdata.c$C.num, center = TRUE, scale = FALSE)
# center binary (or categorical) covariates(s)
# only neccessary if the desired baseline level is NOT the default baseline level.
sdata.c$C.bin <- relevel(sdata.c$C.bin, ref = "1")

#---------------------------------------------------------------------------------#
# Example 1: Continuous Mediator
#---------------------------------------------------------------------------------#
fit.m1 <- lm(M.num ~ R + C.num + C.bin, data = sdata.c)
fit.y1 <- lm(Y.num ~ R + M.num + M.num:R + X +
          C.num + C.bin, data = sdata.c)
res1 <- pocr(fit.m = fit.m1, fit.y = fit.y1, sims = 40,
        covariates = c("C.num", "C.bin"), treat = "R", seed = 111)
res1

#---------------------------------------------------------------------------------#
# Example 2: Binary Mediator
#---------------------------------------------------------------------------------#
\donttest{fit.x1 <- lm(X ~ R + C.num + C.bin, data = sdata.c)
fit.m2 <- glm(M.bin ~ R + C.num + C.bin, data = sdata.c,
          family = binomial(link = "logit"))
fit.y2 <- lm(Y.num ~ R + M.bin + M.bin:R + X +
          C.num + C.bin, data = sdata.c)
res2 <- pocr(fit.x = fit.x1, fit.m = fit.m2, fit.y = fit.y2,
        sims = 40, covariates = c("C.num", "C.bin"), treat = "R", seed = 111)
res2}

#---------------------------------------------------------------------------------#
# Example 3: Ordinal Mediator
#---------------------------------------------------------------------------------#
\donttest{require(MASS)
fit.m3 <- polr(M.cat ~ R + C.num + C.bin, data = sdata.c)
fit.y3 <- lm(Y.num ~ R + M.cat + M.cat:R + X +
	C.num + C.bin, data = sdata.c)
res3 <- pocr(fit.x = fit.x1, fit.m = fit.m3, fit.y = fit.y3,
       sims = 40, covariates = c("C.num", "C.bin"), treat = "R", seed = 111)
res3}

#---------------------------------------------------------------------------------#
# Example 4: Nominal Mediator
#---------------------------------------------------------------------------------#
\donttest{require(nnet)
fit.m4 <- multinom(M.cat ~ R + C.num + C.bin, data = sdata.c)
res4 <- pocr(fit.x = fit.x1, fit.m = fit.m4, fit.y = fit.y3,
        sims = 40, covariates = c("C.num", "C.bin"), treat = "R", seed = 111)
res4}
}
\references{
Park, S., Kang, S., and Lee, C. (2021+). "Choosing an optimal method for causal
decomposition analysis: A better practice for identifying contributing factors to
health disparities". arXiv preprint arXiv:2109.06940.
}
\seealso{
\code{\link{mmi}}, \code{\link{smi}}
}
\author{
Suyeon Kang, University of California, Riverside, \email{skang062@ucr.edu};
Soojin Park, University of California, Riverside, \email{soojinp@ucr.edu}.
}
