% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-distributions.R
\name{MCMC-distributions}
\alias{MCMC-distributions}
\alias{mcmc_dens}
\alias{mcmc_dens_overlay}
\alias{mcmc_hist}
\alias{mcmc_hist_by_chain}
\alias{mcmc_violin}
\title{Histograms and kernel density plots of MCMC draws}
\usage{
mcmc_hist(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., binwidth = NULL)

mcmc_dens(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., trim = FALSE)

mcmc_hist_by_chain(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., binwidth = NULL)

mcmc_dens_overlay(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., trim = FALSE)

mcmc_violin(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., probs = c(0.1, 0.5,
  0.9))
}
\arguments{
\item{x}{A 3-D array, matrix, list of matrices, or data frame of MCMC draws.
The \link{MCMC-overview} page provides details on how to specify each these
allowed inputs.}

\item{pars}{An optional character vector of parameter names. If neither
\code{pars} nor \code{regex_pars} is specified then the default is to use
\emph{all} parameters.}

\item{regex_pars}{An optional \link[=grep]{regular expression} to use for
parameter selection. Can be specified instead of \code{pars} or in addition
to \code{pars}.}

\item{transformations}{An optional named list specifying transformations to
apply to parameters. The name of each list element should be a parameter
name and the content of each list element should be a function (or any item
to match as a function via \code{\link{match.fun}}, e.g. a string naming a
function). If a function in the list of transformations is specified by its
name as a string (e.g. \code{"log"}), then it can be used to construct a
new parameter label for the appropriate parameter (e.g.
\code{"log(sigma)"}). If a function itself is specified (e.g. \code{log} or
\code{function(x) log(x)}) then \code{"t"} is used in the new parameter
label to indicate that the parameter is transformed (e.g.
\code{"t(sigma)"}).}

\item{facet_args}{Arguments (other than \code{facets}) passed to
\code{\link[ggplot2]{facet_wrap}} (if \code{by_chain} is \code{FALSE}) or
\code{\link[ggplot2]{facet_grid}} (if \code{by_chain} is \code{TRUE}) to
control faceting.}

\item{...}{Currently ignored.}

\item{binwidth}{An optional value used as the \code{binwidth} argument to
\code{\link[ggplot2]{geom_histogram}} to override the default binwidth.}

\item{trim}{A logical scalar passed to \code{\link[ggplot2]{geom_density}}.}

\item{probs}{A numeric vector passed to \code{\link[ggplot2]{geom_violin}}'s
\code{draw_quantiles} argument to specify at which quantiles to draw
horizontal lines. Set to \code{NULL} to remove the lines.}
}
\value{
A ggplot object that can be further customized using the
  \pkg{ggplot2} package.
}
\description{
Various types of histograms and kernel density plots of MCMC draws. See the
\strong{Plot Descriptions} section, below, for details.
}
\section{Plot Descriptions}{

\describe{
  \item{\code{mcmc_hist}}{
   Histograms of posterior draws with all chains merged.
  }
  \item{\code{mcmc_dens}}{
   Kernel density plots of posterior draws with all chains merged.
  }
  \item{\code{mcmc_hist_by_chain}}{
   Histograms of posterior draws with chains separated via faceting.
  }
  \item{\code{mcmc_dens_overlay}}{
   Kernel density plots of posterior draws with chains separated but
   overlaid on a single plot.
  }
  \item{\code{mcmc_violin}}{
   The density estimate of each chain is plotted as a violin with
   horizontal lines at notable quantiles.
  }
}
}
\examples{
# some parameter draws to use for demonstration
x <- example_mcmc_draws()
dim(x)
dimnames(x)

##################
### Histograms ###
##################

# histograms of all parameters
color_scheme_set("brightblue")
mcmc_hist(x)

# histograms of some parameters
color_scheme_set("pink")
mcmc_hist(x, pars = c("alpha", "beta[2]"))
mcmc_hist(x, pars = "sigma", regex_pars = "beta")

# example of using 'transformations' argument to plot log(sigma),
# and parsing facet labels (e.g. to get greek letters for parameters)
mcmc_hist(x, transformations = list(sigma = "log"),
          facet_args = list(labeller = ggplot2::label_parsed)) +
          facet_text(size = 15)

# instead of list(sigma = "log"), you could specify the transformation as
# list(sigma = log) or list(sigma = function(x) log(x)), but then the
# label for the transformed sigma is 't(sigma)' instead of 'log(sigma)'
mcmc_hist(x, transformations = list(sigma = log))

# separate histograms by chain
color_scheme_set("pink")
mcmc_hist_by_chain(x, regex_pars = "beta")

#################
### Densities ###
#################

mcmc_dens(x, pars = c("sigma", "beta[2]"),
          facet_args = list(nrow = 2))

# separate and overlay chains
color_scheme_set("mix-teal-pink")
mcmc_dens_overlay(x, pars = c("sigma", "beta[2]"),
                  facet_args = list(nrow = 2)) +
                  facet_text(size = 14)

# separate chains as violin plots
color_scheme_set("green")
mcmc_violin(x) + panel_bg(color = "gray20", size = 2, fill = "gray30")

}
\seealso{
Other MCMC: \code{\link{MCMC-combos}},
  \code{\link{MCMC-diagnostics}},
  \code{\link{MCMC-intervals}}, \code{\link{MCMC-nuts}},
  \code{\link{MCMC-overview}},
  \code{\link{MCMC-scatterplots}},
  \code{\link{MCMC-traces}}
}

