% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/backbone_from_unweighted.R
\name{backbone_from_unweighted}
\alias{backbone_from_unweighted}
\title{Extract the backbone from an unweighted, undirected network}
\usage{
backbone_from_unweighted(
  U,
  model = "lspar",
  parameter = 0.5,
  escore,
  normalize,
  filter,
  umst,
  narrative = FALSE,
  return = "backbone"
)
}
\arguments{
\item{U}{An unweighted, undirected network as an adjacency matrix or \link[Matrix]{Matrix}, or an unweighted unipartite \link[igraph]{igraph} object}

\item{model}{string: backbone model}

\item{parameter}{real: filtering parameter}

\item{escore}{string: Method for scoring edges' importance}

\item{normalize}{string: Method for normalizing edge scores}

\item{filter}{string: Type of filter to apply}

\item{umst}{boolean: TRUE if the backbone should include the union of maximum spanning trees, to ensure connectivity}

\item{narrative}{boolean: display suggested text & citations}

\item{return}{string: return either only the \code{"backbone"} or \code{"everything"}}
}
\value{
If \code{return = "backbone"}, a backbone in the same class as \code{U}. If \code{return = "everything"}, then the backbone
is returned as an element in a list that also includes the original unweighted network, a narrative description, and the original
function call.
}
\description{
\code{backbone_from_unweighted()} extracts the unweighted backbone from an unweighted, undirected network
}
\details{
The \code{backbone_from_unweighted} function extracts the backbone from an unweighted unipartite network. The backbone is an
unweighted unipartite network that contains only edges preserved by a backbone model.

The following backbone models are available using the \code{model} parameter:
\itemize{
\item \code{skeleton} - Skeleton backbone (Karger, 1999)
\item \code{gspar} - Global Sparsification (Satuluri et al., 2011)
\item \code{lspar} - Local Sparsification (Satuluri et al., 2011)
\item \code{simmelian} - Simmelian backbone (Nick et al., 2013)
\item \code{jaccard} - Jaccard backbone (Goldberg and Roth, 2003)
\item \code{meetmin} - MeetMin backbone (Goldberg and Roth, 2003)
\item \code{geometric} - Geometric backbone (Goldberg and Roth, 2003)
\item \code{hyper} - Hypergeometric backbone, (Goldberg and Roth, 2003)
\item \code{degree} - Local Degree backbone (Hamann et al, 2016)
\item \code{quadrilateral} - Quadrilateral Simmelian backbone (Nocaj et al, 2015)
\item \code{custom} - A custom backbone model specified by \code{escore}, \code{normalize}, \code{filter}, and \code{umst}
}

The \code{escore} parameter determines how an unweighted edge's importance is calculated.
\itemize{
\item \code{random}: a random number drawn from a uniform distribution
\item \code{betweenness}: edge betweenness
\item \code{triangles}: number of triangles that include the edge
\item \code{jaccard}: jaccard similarity coefficient of the neighborhoods of an edge's endpoints, or alternatively, triangles normalized by the size of the union of the endpoints neighborhoods
\item \code{dice}: dice similarity coefficient of the neighborhoods of an edge's endpoints
\item \code{quadrangles}: number of quadrangles that include the edge
\item \code{quadrilateral}: geometric mean normalization of quadrangles
\item \code{degree}: degree of neighbor to which an edge is adjacent (asymmetric)
\item \code{meetmin}: triangles normalized by the smaller of the endpoints' neighborhoods' sizes
\item \code{geometric}: triangles normalized by the product of the endpoints' neighborhoods' sizes
\item \code{hypergeometric}: probability of the edge being included at least as many triangles if edges were random, given the size of the endpoints' neighborhoods (inverted, so that larger is more important)
}

The \code{normalize} parameter determines whether edge scores are normalized.
\itemize{
\item \code{none}: no normalization is performed
\item \code{rank}: scores are normalized by neighborhood rank, such that the strongest edge in a node's neighborhood is ranked 1 (requires that \code{filter = degree})
\item \code{embeddedness}: scores are normalized using the maximum Jaccard coefficient of the top k-ranked neighbors of each endpoint, for all k
}

The \code{filter} parameter determines how edges are filtered based on their (normalized) edge scores.
\itemize{
\item \code{threshold}: Edges with scores > \code{parameter} are retained in the backbone
\item \code{proportion}: Specifies the approximate proportion of edges to retain in the backbone
\item \code{degree}: Retains each node's d^\code{parameter} most important edges, where d is the node's degree (requires that \code{normalize = "rank"})
\item \code{disparity}: Applies the disparity filter using \code{\link[=backbone_from_weighted]{backbone_from_weighted()}}
\item \code{lans}: Applies locally adaptive network sparsification using \code{\link[=backbone_from_weighted]{backbone_from_weighted()}}
\item \code{mlf}: Applies the marginal likelihood filter using \code{\link[=backbone_from_weighted]{backbone_from_weighted()}}
}
}
\examples{
#A dense, unweighted network with three embedded communities
U <- igraph::sample_sbm(60, matrix(c(.75,.25,.25,.25,.75,.25,.25,.25,.75),3,3), c(20,20,20))
plot(U)  #Communities are not obvious

#Extract backbone using the built-in "Local Sparsification" model
bb <- backbone_from_unweighted(U, model = "lspar", parameter = 0.5)
plot(bb)  #Communities are clearly visible

#Extract backbone using local sparification, but explicitly specifying the model steps
bb <- backbone_from_unweighted(U, model = "custom", escore = "jaccard",
                               normalize = "rank", filter = "degree",
                               umst = FALSE, parameter = 0.5)
plot(bb)  #Communities are clearly visible
}
\references{
package: {Neal, Z. P. (2025). backbone: An R Package to Extract Network Backbones. CRAN. \doi{10.32614/CRAN.package.backbone}}

skeleton: {Karger, D. R. (1999). Random sampling in cut, flow, and network design problems. \emph{Mathematics of Operations Research, 24}, 383-413. \doi{10.1287/moor.24.2.383}}

gspar and lspar: {Satuluri, V., Parthasarathy, S., & Ruan, Y. (2011, June). Local graph sparsification for scalable clustering. In Proceedings of the 2011 ACM SIGMOD International Conference on Management of data (pp. 721-732). \doi{10.1145/1989323.1989399}}

simmelian: {Nick, B., Lee, C., Cunningham, P., & Brandes, U. (2013, August). Simmelian backbones: Amplifying hidden homophily in facebook networks. In Proceedings of the 2013 IEEE/ACM international conference on advances in social networks analysis and mining (pp. 525-532). \doi{10.1145/2492517.2492569}}

jaccard, meetmin, geometric, hyper: {Goldberg, D. S., & Roth, F. P. (2003). Assessing experimentally derived interactions in a small world. \emph{Proceedings of the National Academy of Sciences, 100}, 4372-4376. \doi{10.1073/pnas.0735871100}}

degree: {Hamann, M., Lindner, G., Meyerhenke, H., Staudt, C. L., & Wagner, D. (2016). Structure-preserving sparsification methods for social networks. \emph{Social Network Analysis and Mining, 6}, 22. \doi{10.1007/s13278-016-0332-2}}

quadrilateral: {Nocaj, A., Ortmann, M., & Brandes, U. (2015). Untangling the hairballs of multi-centered, small-world online social media networks. \emph{Journal of Graph Algorithms and Applications, 19}, 595-618. \doi{10.7155/jgaa.00370}}
}
