#' The stochastic degree sequence model (sdsm)
#'
#' `sdsm` computes the proportion of generated edges
#'     above or below the observed value using the stochastic degree sequence model.
#'     Once computed, use \code{\link{backbone.extract}} to return
#'     the backbone matrix for a given alpha value.
#'
#' @param B Matrix: Bipartite network
#' @param trials Integer: Number of random bipartite graphs generated
#' @param model String: A generalized linear model (glm) used to generate random bipartite graphs.
#' @param sparse Boolean: If sparse matrix manipulations should be used
#' @param maxiter Integer: Maximum number of iterations if "model" is a glm.
#' @param dyad vector length 2: two row entries i,j. Saves each value of the i-th row and j-th column in each projected B* matrix. This is useful for visualizing an example of the empirical null edge weight distribution generated by the model. These correspond to the row and column indices of a cell in the projected matrix , and can be written as their string row names or as numeric values.
#' @param progress Boolean: If \link[utils]{txtProgressBar} should be used to measure progress
#'
#' @details The 'model' parameter can take in a 'link' function, as described by \link[stats]{glm} and \link[stats]{family}. This can be one of c('logit', 'probit', 'cauchit', 'log', 'cloglog').
#' @details During each iteration, sdsm computes a new B* matrix. This is a random bipartite matrix with about the same row and column sums as the original matrix B.
#' If the dyad_parameter is indicated to be used in the parameters, when the B* matrix is projected, the projected value for the corresponding row and column will be saved.
#' This allows the user to see the distribution of the edge weights for desired row and column.
#'
#'
#' @return list(positive, negative, dyad_values).
#' positive: matrix of proportion of times each entry of the projected matrix B is above the corresponding entry in the generated projection.
#' negative: matrix of proportion of times each entry of the projected matrix B is below the corresponding entry in the generated projection.
#' dyad_values: list of edge weight for i,j in each generated projection.
#'
#'
#' @references \href{https://www.sciencedirect.com/science/article/abs/pii/S0378873314000343}{Neal, Z. P. (2014). The backbone of bipartite projections: Inferring relationships from co-authorship, co-sponsorship, co-attendance, and other co-behaviors. Social Networks, 39, Elsevier: 84-97. DOI: 10.1016/j.socnet.2014.06.001}
#'
#'
#' @export
#'
#' @examples
#'sdsm_props <- sdsm(davis, trials = 100,dyad = c("EVELYN", "CHARLOTTE" ))
sdsm <- function(B,
                 trials = 1000,
                 model = "logit",
                 sparse = TRUE,
                 maxiter = 25,
                 dyad = NULL,
                 progress = FALSE){

  #Argument Checks
  if ((sparse!="TRUE") & (sparse!="FALSE")) {stop("sparse must be either TRUE or FALSE")}
  if ((model!="logit") & (model!="probit") & (model!="log") & (model!="cloglog")) {stop("model must be: logit | probit | log | cloglog")}
  if ((trials < 1) | (trials%%1!=0)) {stop("trials must be a positive integer")}
  if (class(B)!="matrix") {stop("input bipartite data must be a matrix")}


  #Project to one-mode data
  if (sparse=="TRUE") {
    B <- Matrix::Matrix(B,sparse=T)
    P<-B%*%Matrix::t(B)
  }
  if (sparse=="FALSE") {P<-B%*%t(B)}

  #Create Positive and Negative Matrices to hold backbone
  Positive <- matrix(0, nrow(P), ncol(P))
  Negative <- matrix(0, nrow(P), ncol(P))

  #Compute probabilities for SDSM (alternative is in star)
  #Vectorize the bipartite data
  A <- data.frame(as.vector(B))
  names(A)[names(A)=="as.vector.B."] <- "value"

  #Assign row and column IDs in the vectorized data
  A$row <- rep(1:nrow(B), times=ncol(B))
  A$col <- rep(1:ncol(B), each=nrow(B))

  #Compute and attach rowsums, columnsums, interact
  A$rowmarg <- stats::ave(A$value,A$row,FUN=sum)
  A$colmarg <- stats::ave(A$value,A$col,FUN=sum)
  A$rowcol<-A$rowmarg*A$colmarg

  #Estimate logit model, compute probabilities
  model.estimates <- stats::glm(formula= value ~  rowmarg + colmarg + rowcol, family = stats::binomial(link=model), data=A, control = list(maxit = maxiter))
  probs <- as.vector(stats::predict(model.estimates,newdata=A,type = "response"))

  #Dyad save
  edge_weights <- numeric(trials)
  if (length(dyad) > 0){
    if (class(dyad[1]) != "numeric"){
      vec <- match(c(dyad[1], dyad[2]), rownames(B))
    }
    else{
      vec <- dyad
    }
  }

  #Build null models
  for (i in 1:trials){

    #Start estimation timer; print message
    if (i == 1) {
      start.time <- Sys.time()
      message("Finding the Backbone using ", model, " SDSM")
    }

    #Use GLM probabilities to create an SDSM Bstar
    #Bstar <- matrix(rbinom(nrow(B) * ncol(B), 1, probs), nrow(B), ncol(B))  #Equivalent, but slightly slower
    Bstar <- matrix(((stats::runif(nrow(B) * ncol(B)))<=probs)+0, nrow(B), ncol(B))
    if (sparse=="TRUE") {Bstar <- Matrix::Matrix(Bstar,sparse=T)}


    #Construct Pstar from Bstar
    if (sparse=="TRUE") {Pstar<-Bstar%*%Matrix::t(Bstar)}
    if (sparse=="FALSE") {Pstar<-Bstar%*%t(Bstar)}

    #Check whether Pstar edge is larger/smaller than P edge
    Positive <- Positive + (Pstar > P)+0
    Negative <- Negative + (Pstar < P)+0

    #Save Dyad of P
    if (length(dyad) > 0){
      edge_weights[i] <- Pstar[vec[1], vec[2]]
    }

    #Report estimated running time, update progress bar
    if (i==10){
      end.time <- Sys.time()
      est = (round(difftime(end.time, start.time), 2) * (trials/10))
      message("Estimated time to complete is ", est," ", units(est))
      if (progress == "TRUE"){
        pb <- utils::txtProgressBar(min = 0, max = trials, style = 3)
      }
    } #end timer estimate

    if ((progress == "TRUE") & (i>=10)) {utils::setTxtProgressBar(pb, i)}
  } #end for loop
  if (progress == "TRUE"){close(pb)}

  #Proporition of greater than expected and less than expected
  Positive <- (Positive/trials)
  Negative <- (Negative/trials)
  rownames(Positive) <- rownames(B)
  colnames(Positive) <- rownames(B)
  rownames(Negative) <- rownames(B)
  colnames(Negative) <- rownames(B)

  #Save Dyad of P
  if (length(dyad) == 0){
    edge_weights <- NULL
  }


  if (length(dyad) > 0){
    return(list(positive = Positive, negative = Negative, dyad_values = edge_weights))
  }

  else {
    return(list(positive = Positive, negative = Negative))
  }

} #end sdsm function

