context("Tests locate_credentials()")

if (file.exists(".aws/credentials")) {
    
    test_that("locate_credentials() returns envvar values when environment variables and credentials file present", {
        
        skip_on_cran()
        
        # save environment variables
        e <- Sys.getenv(c("AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY", "AWS_SESSION_TOKEN", "AWS_DEFAULT_REGION"))
        
        # set environment variables
        Sys.setenv("AWS_ACCESS_KEY_ID" = "foo-key")
        Sys.setenv("AWS_SECRET_ACCESS_KEY" = "foo-secret")
        Sys.setenv("AWS_SESSION_TOKEN" = "foo-token")
        Sys.setenv("AWS_DEFAULT_REGION" = "foo-region")
        
        # tests
        cred <- locate_credentials()
        expect_true(cred[["key"]] == "foo-key")
        expect_true(cred[["secret"]] == "foo-secret")
        expect_true(cred[["session_token"]] == "foo-token")
        expect_true(cred[["region"]] == "foo-region")
        
        # restore environment variables
        do.call("Sys.setenv", as.list(e))
    })
    
    test_that("locate_credentials() returns non-default values if requested when environment variables and credentials file present", {
        
        skip_on_cran()
        
        # save environment variables
        e <- Sys.getenv(c("AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY", "AWS_SESSION_TOKEN", "AWS_DEFAULT_REGION"))
        
        # set environment variables
        Sys.setenv("AWS_ACCESS_KEY_ID" = "foo-key")
        Sys.setenv("AWS_SECRET_ACCESS_KEY" = "foo-secret")
        Sys.setenv("AWS_SESSION_TOKEN" = "foo-token")
        Sys.setenv("AWS_DEFAULT_REGION" = "foo-region")
        
        # tests
        cred <- locate_credentials(key = "foo1", secret = "foo2", session_token = "foo3", region = "foo4")
        expect_true(cred[["key"]] == "foo1", label = "locate_credentials(key = 'foo1')")
        expect_true(cred[["secret"]] == "foo2", label = "locate_credentials(secret = 'foo2')")
        expect_true(cred[["session_token"]] == "foo3", label = "locate_credentials(session_token = 'foo3')")
        expect_true(cred[["region"]] == "foo4", label = "locate_credentials(region = 'foo4')")
        
        # restore environment variables
        do.call("Sys.setenv", as.list(e))
    })
    
    test_that("locate_credentials() returns credentials file values when environment variables missing and credentials file present", {
        
        skip_on_cran()
        
        # save environment variables
        e <- Sys.getenv(c("AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY", "AWS_SESSION_TOKEN", "AWS_DEFAULT_REGION"))
        
        # unset environment variables
        Sys.unsetenv("AWS_ACCESS_KEY_ID")
        Sys.unsetenv("AWS_SECRET_ACCESS_KEY")
        Sys.unsetenv("AWS_SESSION_TOKEN")
        Sys.unsetenv("AWS_DEFAULT_REGION")
        
        # tests
        cred <- locate_credentials()
        expect_true(cred[["key"]] == "ACCESS_KEY")
        expect_true(cred[["secret"]] == "SECRET_KEY")
        expect_true(cred[["session_token"]] == "TOKEN")
        expect_true(cred[["region"]] == "us-east-1")
        
        # restore environment variables
        do.call("Sys.setenv", as.list(e))
    })
    
    test_that("locate_credentials() returns non-default values if requested, when environment variables missing and credentials file absent", {
        
        skip_on_cran()
        
        # move credentials file
        tmp <- tempfile()
        file.rename(".aws/credentials", tmp)
        
        # save environment variables
        e <- Sys.getenv(c("AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY", "AWS_SESSION_TOKEN", "AWS_DEFAULT_REGION"))
        
        # unset environment variables
        Sys.unsetenv("AWS_ACCESS_KEY_ID")
        Sys.unsetenv("AWS_SECRET_ACCESS_KEY")
        Sys.unsetenv("AWS_SESSION_TOKEN")
        Sys.unsetenv("AWS_DEFAULT_REGION")
        
        # tests
        cred <- locate_credentials(key = "foo1", secret = "foo2", session_token = "foo3", region = "foo4")
        expect_true(cred[["key"]] == "foo1", label = "locate_credentials(key = 'foo1')")
        expect_true(cred[["secret"]] == "foo2", label = "locate_credentials(secret = 'foo2')")
        expect_true(cred[["session_token"]] == "foo3", label = "locate_credentials(session_token = 'foo3')")
        expect_true(cred[["region"]] == "foo4", label = "locate_credentials(region = 'foo4')")
        
        # restore credentials file
        file.rename(tmp, ".aws/credentials")
        
        # restore environment variables
        do.call("Sys.setenv", as.list(e))
    })
    
    test_that("locate_credentials() returns NULLs when environment variables missing and credentials file absent", {
        
        skip_on_cran()
        
        # move credentials file
        tmp <- tempfile()
        file.rename(".aws/credentials", tmp)
        
        # save environment variables
        e <- Sys.getenv(c("AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY", "AWS_SESSION_TOKEN", "AWS_DEFAULT_REGION"))
        
        # unset environment variables
        Sys.unsetenv("AWS_ACCESS_KEY_ID")
        Sys.unsetenv("AWS_SECRET_ACCESS_KEY")
        Sys.unsetenv("AWS_SESSION_TOKEN")
        Sys.unsetenv("AWS_DEFAULT_REGION")
        
        # tests
        cred <- locate_credentials()
        expect_true(is.null(cred[["key"]]))
        expect_true(is.null(cred[["secret"]]))
        expect_true(is.null(cred[["session_token"]]))
        expect_true(cred[["region"]] == "us-east-1")
        
        # restore credentials file
        file.rename(tmp, ".aws/credentials")
        
        # restore environment variables
        do.call("Sys.setenv", as.list(e))
    })
    
}
