#' Create an image plot with a legend scale.
#'
#' \code{autoimage} creates an image plot while also automatically plotting a legend that indicates the correspondence between the colors and the values of the \code{z} variable.  \code{autoimage} is intended to be backwards compatible with the \code{image} function, but no promises are made.  Additionally, one can plot multiple images in one graphic using this function, both with and without a common scale for the images.  Perhaps more importantly, the \code{\link[fields]{poly.image}} function from the \code{fields} package is used to provide image plots of data on an irregular grid (e.g., data measured at longitude/latitude coordinates).
#'
#' When \code{project = TRUE}, the \code{\link[mapproj]{mapproject}} function is used to project the \code{x} and \code{y} coordinates.  In that case, the projected \code{x} and \code{y} coordinates of the plotted image are unlikely to be similar to the original values.  It is recommended that the user sets \code{axes = FALSE} since the x and y axis scales will not be interpretable.  However, the axes will still be useful for scaling purposes using \code{xlim} and \code{ylim}.  If reference axes are still desired, set \code{map.grid = TRUE}, in which case the \code{\link[mapproj]{map.grid}} function is used to draw correct longitude and latitude grid lines.
#'
#' If multiple images are to be plotted (i.e., if \code{z} is an array), then the \code{main} argument can be a vector with length matching \code{dim(z)[3]}, and each successive element of the vector will be used to add a title to each successive image plotted.  See the Examples.
#'
#' Note that the more images that are plotted simulataneously, the smaller one typically wants \code{mratio} to be.
#'
#' The multiple plots are constructed using the \code{\link[graphics]{layout}} function, which is incompatible with the \code{mfrow} and \code{mfcol} arguments in the \code{\link[graphics]{par}} function and is also incompatible with the \code{\link[graphics]{split.screen}} function.
#'
#' @param x,y Locations of grid lines at which the values in \code{z} are measured.  The values must be finite and non-missing.  These arguments can be either vectors, in which case the values must be in (strictly) ascending order.  If an irregular grid is to be used, then \code{x} and \code{y} should be numeric matrices having the same number of rows and columns as \code{z}.  If these arguments are not provided, equally spaced values from 0 to 1 are used by default. If \code{x} is a list, its components \code{x$x} and \code{x$y} are used for \code{x} and \code{y}, respectively. If the list has component \code{x$z}, this is used for \code{z}.
#' @param z A numeric or logical matrix containing the values to be plotted (NAs are allowed). If multiple images are to be plotted, a numeric array can be provided instead.  The third dimension of the array indicates the number of images that should be plotted.  Note that \code{x} can be used instead of \code{z} for convenience.
#' @param col A list of colors such as that generated by \code{\link[grDevices]{rainbow}}, \code{\link[grDevices]{heat.colors}}, \code{\link[grDevices]{topo.colors}}, \code{\link[grDevices]{terrain.colors}} or similar functions.  The default is given by \code{heat.colors(12)}.
#' @param legend A logical value indicating whether a legend scale should be added to the plot.  Default is \code{TRUE}.
#' @param horizontal A logical value indicating whether the legend scale should be horizontal or vertical.  Default is \code{TRUE}, indicating the legend scale will be on the bottom of the plot and the colors will run from left to right.  If \code{FALSE}, the legend scale will be on the right side of the plot and the colors will run vertically.
#' @param common.legend A logical value indicating whether a common legend scale should be used for all images provided in the \code{z} array.  Default is \code{TRUE}.
#' @param size A vector of length two indicating the number of rows and columns that should be used for the series of image data in \code{z}.  Note that \code{prod(size)} must match the length of the third dimension of \code{z} (if it is an array), or \code{c(1, 1)} if \code{z} is a matrix.
#' @param mratio A numeric value indicating the ratio of the width of each image to the width of the legend scale.  Default is \code{mratio = 5}.
#' @param project A logical value indicating whether the \code{x} and \code{y} coordinates should be projected before plotting.  This only makes sense if the values in \code{x} and \code{y} are longitude and latitude coordinates.  Default is \code{FALSE}.  If \code{TRUE}, then the \code{\link[mapproj]{mapproject}} function is used to project the coordinates.  If \code{TRUE}, then additional arguments for \code{\link[mapproj]{mapproject}} should be provided in \code{project.args}.
#' @param map.grid A logical value indicating whether a map grid should be added to the current plot using the \code{\link[mapproj]{map.grid}} function.  This will only be sensible when \code{project = TRUE}.  See Details.  Additional arguments to \code{\link[mapproj]{map.grid}} can be provided as a list to \code{grid.args}.
#' @param mmar A vector of length four indicating the margins of each image in the plot.  This is passed internally to the \code{mar} argument of the \code{\link[graphics]{par}} function.  See the \code{\link[graphics]{par}} function for more details.  The default is the currently specified \code{mar} argument of the \code{par} function.
#' @param legend.mar A vector of length four indicating the margins of the legend scale.  This is passed internally to the \code{mar} argument of the \code{\link[graphics]{par}} function.  See the \code{\link[graphics]{par}} function for more details.  The default is sensible values based on the value of \code{mmar}.
#' @param axis.args A list with arguments matching the arguments of the \code{\link[graphics]{axis}} function that is used to customize the legend scale.  See \code{\link{legend.scale}} for examples.
#' @param project.args A list with arguments matching the non \code{x} and \code{y} arguments of the \code{\link[mapproj]{mapproject}} function.
#' @param grid.args A list with arguments matching the non \code{lim} arguments of the \code{\link[mapproj]{map.grid}} function.  This is used to customize the plotted grid when \code{map.grid = TRUE}.
#' @param map.poly A list with named elements \code{x} and \code{y} that are used to plot relevent polygons on each image using the \code{\link[graphics]{lines}} function.
#' @param poly.args A list with arguments matching those provided to the \code{\link[graphics]{lines}} function used to plot \code{map.poly}.  This would be used to customize the lines, e.g., with different thickness, type, or color.
#' @param ... Additional arguments passed to the \code{\link[graphics]{image}} or \code{\link[fields]{poly.image}} functions.  e.g., \code{xlab}, \code{ylab}, \code{xlim}, \code{ylim}, \code{zlim}, etc.
#' @references The code for this function is derived from the internals of the \code{\link[fields]{image.plot}} function written by Doug Nychka and from \code{image.scale.2} function written by Marc Taylor and discussed at \code{http://menugget.blogspot.com/2013/12/new-version-of-imagescale-function.html}.
#' @seealso \code{\link[graphics]{image}}, \code{\link[fields]{image.plot}}, \code{\link[graphics]{axis}}
#' @return NULL
#' @importFrom fields poly.image
#' @importFrom mapproj map.grid mapproject
#' @importFrom graphics axTicks axis box image layout par
#' @examples
#' # Example from image function documentation
#' x <- y <- seq(-4*pi, 4*pi, len = 27)
#' r <- sqrt(outer(x^2, y^2, "+"))
#' z <- cos(r^2)*exp(-r/6)
#' image(z, col  = gray((0:32)/32))
#' autoimage(z, col  = gray((0:32)/32), legend = FALSE)
#'
#' # now with legend
#' autoimage(z, col  = gray((0:32)/32))
#' autoimage(z, col  = gray((0:32)/32), horizontal = FALSE)
#'
#' # add some customization
#' autoimage(x, y, z, xlab = "x1", ylab = "y1", main = "Math is beautiful ...")
#'
#' # now do some examples with multiple images
#' z2 <- cos(r^2/2)*exp(-r/3)
#' z3 <- cos(r^2/2)*exp(-r/6)
#' z4 <- cos(r^2/3)*exp(-r/5)
#' z5 <- cos(r^2/4)*exp(-r/2)
#' z6 <- cos(r^2/5)*exp(-r)
#' library(abind)
#' zarray <- abind(z, z2, z3, z4, z5, z6, along = 3)
#'
#' # multiple images with common scale, separate titles
#' autoimage(x, y, zarray, main = letters[1:6], size = c(3, 2),
#'           mratio = 2, mmar = c(2.1, 4.1, 2.1, 2.1))
#' # change the orientation of the scale
#' autoimage(x, y, zarray, main = letters[1:6], size = c(3, 2),
#'           mratio = 3, mmar = c(4.1, 4.1, 2.1, 2.1),
#'           horizontal = FALSE)
#'
#' # do some examples with an irregular grid
#' # load data from fields package
#' data("RCMexample", package = "fields")
#' # restructure data with 2, 4, and 6 images
#' RCMexample2 = RCMexample
#' RCMexample2$z = RCMexample2$z[,,1:2]
#' RCMexample4 = RCMexample
#' RCMexample4$z = RCMexample4$z[,,1:4]
#'
#' # plot irregularly gridded images
#' autoimage(RCMexample2, col = fields::tim.colors(12), size = c(2, 1),
#'           mmar = c(2.1, 2.1, 1, 2.1), horizontal = FALSE)
#' autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2))
#'
#' # Do the same plot, but with a projection.
#' # Notice that the axes scales seem off because of the projection
#' autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2),
#'           project = TRUE,
#'           project.args = list(projection = "albers", parameters = c(33, 45)))
#' # compare the axes for the projected coordinates to the correct references lines using map.grid.
#' autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2),
#'           project = TRUE, map.grid = TRUE,
#'           project.args = list(projection = "albers", parameters = c(33, 45)),
#'           grid.args = list(col = "yellow", nx = 5, ny = 5))
#'
#' # more images in a plot
#' autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2), horizontal = FALSE, mratio = 3)
#' autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2), mratio = 4)
#' # adjusting the x axis spacing (without using a projection)
#' autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2),
#'           xaxp = c(-160, -40, 8), mratio = 4)
#'
#' # add a nice polygon to the images
#' library(maps) # need to get world map
#' # get the polygon for the world from the maps package
#' worldpoly = map("world", plot = FALSE)
#' # project and plot two images, no axes,
#' # with polygon of national boundaries and map.grid.
#' # This plot is a bit busy
#' autoimage(RCMexample2, size = c(1, 2), project = TRUE, map.grid = TRUE,
#'           project.args = list(projection = "albers", parameters = c(33, 45)),
#'           map.poly = worldpoly, axes = FALSE,
#'           mmar = c(0.5, 0.5, 0.5, 0.5), legend.mar = c(2, 0.5, 0.5, 0.5))
#'
#' # some plots without a common legend
#' autoimage(RCMexample2, size = c(1, 2), project = TRUE,
#'           map.grid = TRUE, axes = FALSE, common.legend = FALSE,
#'           horizontal = FALSE, legend.mar = c(5.1, 0, 4.1, 3.1),
#'           project.args = list(projection = "albers", parameters = c(33, 45)),
#'           map.poly = worldpoly, mratio = 4)
#' autoimage(RCMexample2, size = c(1, 2), project = TRUE, map.grid = TRUE,
#'           project.args = list(projection = "albers", parameters = c(33, 45)),
#'           map.poly = worldpoly, axes = FALSE, common.legend = FALSE,
#'           legend.mar = c(3.1, 4.1, 0, 2.1))
#' @export
autoimage = function(x, y, z, col = heat.colors(12), legend = TRUE, horizontal = TRUE, common.legend = TRUE, size = c(1, 1), mratio = 5, project = FALSE, map.grid = FALSE, mmar, legend.mar, axis.args, project.args, grid.args, map.poly, poly.args, ...){
  # obtain elements of ...
  args = list(...)

  # sort out x, y, and z.  This is mostly identical to the beginning of
  # graphics::image
  if (missing(z)) {
    if (!missing(x)) {
      if (is.list(x)) {
        z <- x$z; y <- x$y; x <- x$x
      } else {
        if(is.null(dim(x)))
          stop("argument must be matrix-like")
        z <- x
        x <- seq.int(0, 1, length.out = nrow(z))
        if(missing(y)) y <- seq.int(0, 1, length.out = ncol(z))
      }
      if (is.null(args$xlab)) args$xlab <- ""
      if (is.null(args$ylab)) args$ylab <- ""
    } else stop("no 'z' matrix specified")
  } else if (is.list(x)) {
    y <- x$y
    x <- x$x
  }

  # convert matrix to array, if z is a matrix, double check size argument
  if(length(dim(z)) == 2){
    if(!all.equal(size, c(1, 1)) & dim(z)[3] == 1){
      stop("size must equal c(1, 1) when z is a matrix")
    }
    dim(z) = c(dim(z), 1)
  }

  # get x and y labels
  if(is.null(args$xlab)){
    args$xlab <- if (missing(x)) "" else deparse(substitute(x))
  }
  if(is.null(args$ylab)){
    args$ylab <- if (missing(y)) "" else deparse(substitute(y))
  }
  # set default arguments for missing arguments
  if(missing(mmar)) mmar = par()$mar
  if(missing(axis.args)) axis.args = list()
  if(is.null(axis.args$las)) axis.args$las = ifelse(horizontal, 0, 2)
  if(missing(legend.mar)){
    legend.mar = mmar
    if(horizontal){
      legend.mar[3] = 0
      legend.mar[1] = 3.1
    }else{
      legend.mar[2] = 0
      legend.mar[4] = 3.1
    }
  }

  # double check that dimensions of size and z match
  if(dim(z)[3] != prod(size)){
    stop("dim(z)[3] != prod(size)")
  }

  # update main to ensure that dimensions match
  ng = prod(size)
  if(!is.null(args$main)){
    if(length(args$main) != ng){
      stop("The number of elements in main must equal prod(size) so that each plot in the grid has a specified title")
    }
  }else{
    args$main = rep("", ng)
  }

  # specify zlims to match dimensions of z depending on whether
  # a common legend will be used
  if(is.null(args$zlim)){
    zl = vector("list", ng)
    if(common.legend){
      ran = range(z, na.rm = TRUE)
      for(j in seq_along(zl)) zl[[j]] = ran
    }else{
      for(j in seq_along(zl)) zl[[j]] = range(z[,,j], na.rm = TRUE)
    }
    args$zlim = zl
  }else{
    if(!is.list(args$zlim)){
      if(length(args$zlim) != 2) stop("If using a common legend, then zlim should be a vector of length 2")
      zl = vector("list", ng)
      for(j in seq_along(zl)) zl[[j]] = args$zlim
      args$zlim = zl
    }else{
      if(length(args$zlim) != ng) stop("zlim should be a list of length prod(size), with each element specifying the zlim of the corresponding image")
    }
  }

  if(is.null(args$axes)) args$axes = TRUE

  if(missing(grid.args)) grid.args = list()
  if(missing(map.poly)) map.poly = NULL
  if(!is.null(map.poly)){
    if(!is.list(map.poly)){
      stop("map.poly must be a list with x and y components")
    }else{
      if(is.null(map.poly$x)) stop("The x component of map.poly is missing")
      if(is.null(map.poly$y)) stop("The y component of map.poly is missing")
    }
  }

  if(project){
    # args$axes = FALSE
    if(is.null(project.args)){
      project.args = list()
    }
    if(!is.matrix(x)){
      x = matrix(x, nrow = dim(z)[1], ncol = dim(z)[2], byrow = TRUE)
    }
    if(!is.matrix(y)){
      y = matrix(y, nrow = dim(z)[1], ncol = dim(z)[2])
    }

    if(map.grid){
      grid.args$lim = c(range(x), range(y))
    }
    if(missing(map.poly)) map.poly = NULL

    if(is.null(project.args$projection)) project.args$project = ""
    projectxy = mapproj::mapproject(c(x), c(y), projection = project.args$projection, parameters = project.args$parameters, orientation = project.args$orientation)
    x = matrix(projectxy$x, nrow = nrow(x))
    y = matrix(projectxy$y, nrow = nrow(y))
    if(!is.null(map.poly)){
      projectpoly = mapproj::mapproject(map.poly$x, map.poly$y)
      map.poly$x = projectpoly$x
      map.poly$y = projectpoly$y
    }
  }
  if(missing(poly.args)) poly.args = list()
  if(!is.null(map.poly)){
    poly.args$x = map.poly$x
    poly.args$y = map.poly$y
  }

  # is the grid a regular grid
  regular = ifelse(length(x) != nrow(z), FALSE, TRUE)
  # decide plotting function accordingly
  plotf = fields::poly.image
  if(regular) plotf = image

  nr = size[1]
  nc = size[2]
  # choose layout depending on whether a legend is required, and if so,
  # whether the legend should be horizontal or vertical and whether the legend
  # is common
  if(!legend){ # setup if there is no legend
    mat = matrix(seq_len(ng), nrow = nr, byrow = TRUE)
    lheight = rep(1, nr)
    lwidth = rep(1, nc)
  }else{ # setup if there should be a legend
    if(common.legend){ # setup the legend is common
      mat = matrix(seq_len(ng), nrow = nr, byrow = TRUE)
      ni = ng + 1
      if(horizontal){
        mat = rbind(mat, matrix(ni, ncol = nc))
        lheight = c(rep(mratio, nr), 1)
        lwidth = c(rep(1, nc))
      }else{
        mat = cbind(mat, matrix(ni, nrow = nr))
        lheight = rep(1, nr)
        lwidth = c(rep(mratio, nc), 1)
      }
    }else{ # setup if the legend is not common
      if(horizontal){ # horizontal legend
        mat = matrix(0, nrow = 2 * nr, ncol = nc)
        for(i in seq_len(nr)){
          crow = (i-1)*2 + 1
          mat[crow, ] = (i - 1)*nc*2 + seq_len(nc)*2 - 1
          mat[crow + 1, ] = (i - 1)*nc*2 + seq_len(nc)*2
        }
        lheight = c(rep(c(mratio, 1), nr))
        lwidth = c(rep(1, nc))
      }else{ # vertical legend
        mat = matrix(1:(2*ng), nrow = nr, ncol = 2*nc, byrow = TRUE)
        lheight = rep(1, nr)
        lwidth = c(rep(c(mratio, 1), nc))
      }
    }
  }

  curpar = par(no.readonly = TRUE)
  layout(mat, heights = lheight, widths = lwidth)
  allmain = args$main
  allzlim = args$zlim
  allaxes = args$axes
  for(i in seq_len(ng)){
    if(regular){
      plotargs = list(x = list(x = x, y = y, z = z[,,i]), col = col)
    }else{
      plotargs = list(x = x, y = y, z = z[,,i], col = col)
    }
    plotargs = c(plotargs, args)
    plotargs$main = allmain[i]
    plotargs$zlim = allzlim[[i]]
    plotargs$axes = FALSE
    if(project) plotargs$asp = 1
    par(mar = mmar)
    do.call(plotf, plotargs)
    if(!is.null(map.poly)) do.call(graphics::lines, poly.args)

    if(allaxes){
      xticks = axTicks(1)
      yticks = axTicks(2)
      xlabels = TRUE
      ylabels = TRUE
      axis(1, at = xticks, labels = xlabels)
      axis(2, at = yticks, labels = ylabels)
      box()
    }
    if(project & map.grid){
      do.call(mapproj::map.grid, grid.args)
    }
    if(legend & !common.legend){
      par(mar = legend.mar)
      legend.scale(zlim = args$zlim[[i]], col = col, horizontal = horizontal, axis.args = axis.args)
    }
  }
  if(legend & common.legend){
    par(mar = legend.mar)
    legend.scale(zlim = args$zlim[[1]], col = col, horizontal = horizontal, axis.args = axis.args)
  }
  par(curpar)
}
