% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CBA_ruleset.R, R/predict.R
\name{CBA_ruleset}
\alias{CBA_ruleset}
\alias{CBA.object}
\alias{predict.CBA}
\alias{print.CBA}
\alias{rules}
\alias{rules.CBA}
\title{Constructor for Objects for Classifiers Based on Association Rules}
\usage{
CBA_ruleset(
  formula,
  rules,
  default = NA,
  method = "first",
  weights = NULL,
  bias = NULL,
  model = NULL,
  discretization = NULL,
  description = "Custom rule set",
  ...
)

rules(x)

\method{rules}{CBA}(x)

\method{predict}{CBA}(object, newdata, type = c("class", "score"), ...)
}
\arguments{
\item{formula}{A symbolic description of the model to be fitted. Has to be
of form \code{class ~ .}. The class is the variable name (part of the item
label before \code{=}).}

\item{rules}{A set of class association rules mined with \code{mineCars} or
\code{apriori} (from \pkg{arules}).}

\item{default}{Default class. If not
specified then objects that are not matched by rules are classified as \code{NA}.}

\item{method}{Classification method \code{"first"} found rule or
\code{"majority"}.}

\item{weights}{Rule weights for method majority. Either a quality measure
available in \code{rules} or a numeric vector of the same length are
\code{rules} can be specified. If missing, then equal weights are used}

\item{bias}{Class bias vector.}

\item{model}{An optional list with model information (e.g., parameters).}

\item{discretization}{A list with discretization information used by \code{predict} to discretize data supplied as a \code{data.frame}.}

\item{description}{Description field used when the classifier is printed.}

\item{\dots}{Additional arguments added as list elements to the CBA object.}

\item{x, object}{An object of class \code{CBA}.}

\item{newdata}{A data.frame or transactions containing rows of new entries
to be classified.}

\item{type}{Predict \code{"class"} labels. Some classifiers can also return
\code{"scores"}.}
}
\value{
\code{CBA_ruleset()} returns an object of class \code{CBA}
representing the trained classifier with fields:
  \item{formula}{used formula.}
  \item{rules}{the classifier rule base.}
  \item{default}{default class label or \code{NA}.}
  \item{method}{classification method.}
  \item{weights}{rule weights.}
  \item{bias}{class bias vector if available.}
  \item{model}{list with model description.}
  \item{discretization}{discretization information.}
  \item{description}{description in human readable form.}

\code{predict} returns predicted labels for \code{newdata}.

\code{rules} returns the rule base.
}
\description{
Objects for classifiers based on association rules have class \code{"CBA"}.
A creator function \code{CBA_ruleset()} and several methods are provided.
}
\details{
\code{CBA_ruleset} creates a new object of class \code{CBA} using the
provides rules as the rule base.  For method \code{"first"}, the user needs
to make sure that the rules are predictive and sorted from most to least
predictive.
}
\examples{

data("iris")

# discretize and create transactions
iris.disc <- discretizeDF.supervised(Species ~., iris)
trans <- as(iris.disc, "transactions")

# create rule base with CARs
cars <- mineCARs(Species ~ ., trans, parameter = list(support = .01, confidence = .8))

cars <- cars[!is.redundant(cars)]
cars <- sort(cars, by = "conf")

# create classifier and use the majority class as the default if no rule matches.
cl <- CBA_ruleset(Species ~ ., cars, method = "first",
  default = uncoveredMajorityClass(Species ~ ., trans, cars))
cl

# look at the rule base
rules(cl)

# make predictions
prediction <- predict(cl, trans)
table(prediction, response(Species ~ ., trans))

# use weighted majority
cl <- CBA_ruleset(Species ~ ., cars, method = "majority", weights = "lift",
  default = uncoveredMajorityClass(Species ~ ., trans, cars))
cl

prediction <- predict(cl, trans)
table(prediction, response(Species ~ ., trans))

}
\seealso{
\code{\link{CBA}}, \code{\link{mineCARs}},
\code{\link[arules]{apriori}}, \code{\link[arules]{rules}},
\code{\link[arules]{transactions}}.
}
\author{
Michael Hahsler
}
