
#' Create a apexcharts.js widget
#'
#' @param ax_opts A \code{list} in JSON format with chart parameters.
#' @param auto_update In Shiny application, update existing chart rather than generating new one.
#' @param width A numeric input in pixels.
#' @param height A numeric input in pixels.
#' @param elementId Use an explicit element ID for the widget.
#'
#' @return A \code{apexcharts} \code{htmlwidget} object.
#' @export
#'
#' @importFrom htmlwidgets createWidget sizingPolicy
#'
#' @example examples/apexchart.R
apexchart <- function(ax_opts = list(), auto_update = TRUE, width = NULL, height = NULL, elementId = NULL) {
  
  # forward options using x
  x <- list(
    ax_opts = ax_opts,
    auto_update = auto_update
  )

  # create widget
  htmlwidgets::createWidget(
    name = 'apexcharter',
    x = x,
    width = width,
    height = height,
    package = 'apexcharter',
    elementId = elementId,
    preRenderHook = function(widget) {
      if (!is.null(widget$x$ax_opts$chart$defaultLocale)) {
        defaultLocale <- widget$x$ax_opts$chart$defaultLocale
        defaultLocale <- match.arg(
          arg = defaultLocale,
          choices = c("de", "el", "en", "es", "fr", "hi", "hr", "hy", "id", "it", 
                      "ko.js", "pt-br", "ru", "tr", "ua")
        )
        if (!is.null(widget$x$ax_opts$chart$locales)) {
          warning("defaultLocale is used but will be ignored since a custom array for locales is provided.")
        } else {
          path <- system.file(file.path("htmlwidgets/lib/apexcharts-locales", paste0(defaultLocale, ".json")), package = "apexcharter")
          locale <- jsonlite::fromJSON(txt = path)
          widget$x$ax_opts$chart$locales <- list(locale)
        }
      }
      widget
    },
    sizingPolicy = htmlwidgets::sizingPolicy(
      defaultWidth = "100%",
      defaultHeight = "100%",
      viewer.defaultHeight = "100%",
      viewer.defaultWidth = "100%",
      knitr.figure = FALSE,
      knitr.defaultWidth = "100%",
      knitr.defaultHeight = "350px",
      browser.fill = TRUE,
      viewer.suppress = FALSE,
      browser.external = TRUE,
      padding = 20
    )
  )
}

#' @title Shiny bindings for apexcharter
#'
#' @description Output and render functions for using apexcharter within Shiny
#' applications and interactive Rmd documents.
#'
#' @param outputId output variable to read from
#' @param width,height Must be a valid CSS unit (like \code{'100\%'},
#'   \code{'400px'}, \code{'auto'}) or a number, which will be coerced to a
#'   string and have \code{'px'} appended.
#' @param expr An expression that generates a apexcharter
#' @param env The environment in which to evaluate \code{expr}.
#' @param quoted Is \code{expr} a quoted expression (with \code{quote()})? This
#'   is useful if you want to save an expression in a variable.
#'   
#' @return An Apexchart output that can be included in the application UI.
#'
#' @name apexcharter-shiny
#'
#' @export
#'
#' @importFrom htmlwidgets shinyWidgetOutput shinyRenderWidget
#' 
#' @example examples/apexcharter-shiny.R
apexchartOutput <- function(outputId, width = '100%', height = '400px'){
  htmlwidgets::shinyWidgetOutput(outputId, 'apexcharter', width, height, package = 'apexcharter')
}

#' @rdname apexcharter-shiny
#' @export
renderApexchart <- function(expr, env = parent.frame(), quoted = FALSE) {
  if (!quoted) { expr <- substitute(expr) } # force quoted
  htmlwidgets::shinyRenderWidget(expr, apexchartOutput, env, quoted = TRUE)
}
