% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/yatchew_test.R
\name{yatchew_test.data.frame}
\alias{yatchew_test.data.frame}
\title{General yatchew_test method for unclassed dataframes}
\usage{
\method{yatchew_test}{data.frame}(data, Y, D, het_robust = FALSE, path_plot = FALSE, ...)
}
\arguments{
\item{data}{(data.frame) A dataframe.}

\item{Y}{(char) Dependent variable.}

\item{D}{(char) Independent variable.}

\item{het_robust}{(logical) If FALSE, the test is performed under the assumption of homoskedasticity (Yatchew, 1997). If TRUE, the test is performed using the heteroskedasticity-robust test statistic proposed by de Chaisemartin, D'Haultfoeuille and Gurgand (2024).}

\item{path_plot}{(logical) if TRUE and \code{D} has length 2, the assigned object will include a plot of the sequence of \eqn{(D_{1i}, D_{2i})}s that minimizes the euclidean distance between each pair of consecutive observations (see Overview for further details).}

\item{...}{Undocumented.}
}
\value{
A list with test results.
}
\description{
General yatchew_test method for unclassed dataframes
}
\section{Overview}{

This program implements the linearity test proposed by Yatchew (1997) and its heteroskedasticity-robust version proposed by de Chaisemartin, D'Haultfoeuille & Gurgand (2024).
In this overview, we sketch the intuition behind the two tests, as to motivate the use of the package and its options.
Please refer to Yatchew (1997) and Section 3 of de Chaisemartin, D'Haultfoeuille & Gurgand (2024) for further details.

Yatchew (1997) proposes a useful extension of the test with multiple independent variables.
The program implements this extension when the \code{D} argument has length \eqn{> 1}.
It should be noted that the power and consistency of the test in the multivariate case are not backed by proven theoretical results.
We implemented this extension to allow for testing and exploratory research.
Future theoretical exploration of the multivariate test will depend on the demand and usage of the package.
\subsection{Univariate Yatchew Test}{

Let \eqn{Y} and \eqn{D} be two random variables.
Let \eqn{m(D) = E[Y|D]}.
The null hypothesis of the test is that \eqn{m(D) = \alpha_0 + \alpha_1 D} for two real numbers \eqn{\alpha_0} and \eqn{\alpha_1}.
This means that, under the null, \eqn{m(.)} is linear in \eqn{D}.
The outcome variable can be decomposed as \eqn{Y = m(D) + \varepsilon}, with \eqn{E[\varepsilon|D] = 0} and \eqn{\Delta Y = \Delta \varepsilon} for \eqn{\Delta D \to 0}.
In a dataset with \eqn{N} i.i.d. realisations of \eqn{(Y, D)}, one can test this hypothesis as follows:
\enumerate{
\item sort the dataset by \eqn{D};
\item denote the corresponding observations by \eqn{(Y_{(i)}, D_{(i)})}, with \eqn{i \in \{1, ..., N\}};
\item approximate \eqn{\hat{\sigma}^2_{\text{diff}}}, i.e. the variance of the first differenced residuals \eqn{\varepsilon_{(i)} - \varepsilon_{(i-1)}}, by the variance of \eqn{Y_{(i)} - Y_{(i-1)}};
\item compute \eqn{\hat{\sigma}^2_{\text{lin}}}, i.e. the variance of the residuals from an OLS regression of \eqn{Y} on \eqn{D}.
}

Heuristically, the validity of step (3) derives from the fact that \eqn{Y_{(i)} - Y_{(i-1)}} = \eqn{m(D_{(i)}) - m(D_{(i-1)})} + \eqn{\varepsilon_{(i)} - \varepsilon_{(i-1)}} and the first difference term is close to zero for \eqn{D_{(i)} \approx D_{(i-1)}}.
Sorting at step (1) ensures that consecutive \eqn{D_{(i)}}s are as close as possible, and when the sample size goes to infinity the distance between consecutive observations goes to zero.
Then, Yatchew (1997) shows that under homoskedasticity and regularity conditions

\deqn{T := \sqrt{G}\left(\dfrac{\hat{\sigma}^2_{\text{lin}}}{\hat{\sigma}^2_{\text{diff}}}-1\right) \stackrel{d}{\longrightarrow} \mathcal{N}\left(0,1\right)}

Then, one can reject the linearity of \eqn{m(.)} with significance level \eqn{\alpha} if \eqn{T > \Phi(1-\alpha)}.

If the homoskedasticity assumption fails, this test leads to overrejection.
De Chaisemartin, D'Haultfoeuille and Gurgand (2024) propose a heteroskedasticity-robust version of the test statistic above.
This version of the Yatchew (1997) test can be implemented by running the command with the option \code{het_robust = TRUE}.
}

\subsection{Multivariate Yatchew Test}{

Let \eqn{\textbf{D}} be a vector of \eqn{K} random variables.
Let \eqn{g(\textbf{D}) = E[Y|\textbf{D}]}.
Denote with \eqn{||.,.||} the Euclidean distance between two vectors.
The null hypothesis of the multivariate test is \eqn{g(\textbf{D}) = \alpha_0 + A'\textbf{D}}, with \eqn{A = (\alpha_1,..., \alpha_K)}, for \eqn{K+1} real numbers \eqn{\alpha_0}, \eqn{\alpha_1}, ..., \eqn{\alpha_K}.
This means that, under the null, \eqn{g(.)} is linear in \eqn{\textbf{D}}.
Following the same logic as the univariate case, in a dataset with \eqn{N} i.i.d. realisations of \eqn{(Y, \textbf{D})} we can approximate the first difference \eqn{\Delta \varepsilon} by \eqn{\Delta Y} valuing \eqn{g(.)} between consecutive observations.
The program runs a nearest neighbor algorithm to find the sequence of observations such that the Euclidean distance between consecutive positions is minimized.
The algorithm has been programmed in C++ and it has been integrated in R thanks to the \code{Rcpp} library. The program follows a very simple nearest neighbor approach:
\enumerate{
\item collect all the Euclidean distances between all the possible unique pairs of rows in \eqn{\textbf{D}} in the matrix \eqn{M}, where \eqn{M_{n,m} = ||\textbf{D}_n,\textbf{D}_m||} with \eqn{n,m \in \{1, ..., N\}};
\item setup the queue to \eqn{Q = \{1, ..., N\}}, the (empty) path vector \eqn{I = \{\}} and the starting index \eqn{i = 1};
\item remove \eqn{i} from \eqn{Q} and find the column index \eqn{j} of M such that \eqn{M_{i,j} = \min_{c \in Q} M_{i,c}};
\item append \eqn{j} to \eqn{I} and start again from step 3 with \eqn{i = j} until \eqn{Q} is empty.
}

To improve efficiency, the program collects only the \eqn{N(N-1)/2} Euclidean distances corresponding to the lower triangle of matrix \eqn{M} and chooses \eqn{j} such that \eqn{M_{i,j} = \min_{c \in Q} 1\{c < i\} M_{i,c} + 1\{c > i\} M_{c,i}}.
The output of the algorithm, i.e. the vector \eqn{I}, is a sequence of row numbers such that the distance between the corresponding rows \eqn{\textbf{D}_i}s is minimized.
The program also uses two refinements suggested in Appendix A of Yatchew (1997):
\itemize{
\item The entries in \eqn{\textbf{D}} are normalized in \eqn{[0,1]};
\item The algorithm is applied to sub-cubes, i.e. partitions of the \eqn{[0,1]^K} space, and the full path is obtained by joining the extrema of the subpaths.
}

By convention, the program computes \eqn{(2\lceil \log_{10} N \rceil)^K} subcubes, where each univariate partition is defined by grouping observations in \eqn{2\lceil \log_{10} N \rceil} quantile bins. If \eqn{K = 2}, the user can visualize in a ggplot graph the exact path across the normalized \eqn{\textbf{D}_i}s by running the command with the option \code{path_plot = TRUE}.

Once the dataset is sorted by \eqn{I}, the program resumes from step (2) of the univariate case.
}
}

\examples{
df <- as.data.frame(matrix(NA, nrow = 1E3, ncol = 0))
df$x <- rnorm(1E3)
df$b <- runif(1E3)
df$y <- 2 + df$b * df$x
yatchew_test(data = df, Y = "y", D = "x")
}
\references{
de Chaisemartin, C., d'Haultfoeuille, X., Gurgand, M. (2024). Two-way Fixed Effects and Difference-in-Difference Estimators in Heterogeneous Adoption Designs.

Yatchew, A. (1997). An elementary estimator of the partial linear model.
}
