% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/xSubneterSNPs.r
\name{xSubneterSNPs}
\alias{xSubneterSNPs}
\title{Function to identify a gene network from an input network given a list of seed SNPs together with the significance level (e.g. GWAS reported p-values)}
\usage{
xSubneterSNPs(data, include.LD = NA, LD.r2 = 0.8,
network = c("STRING_highest", "STRING_high", "STRING_medium",
"PCommonsUN_high", "PCommonsUN_medium", "PCommonsDN_high",
"PCommonsDN_medium", "PCommonsDN_Reactome", "PCommonsDN_KEGG",
"PCommonsDN_HumanCyc", "PCommonsDN_PID", "PCommonsDN_PANTHER",
"PCommonsDN_ReconX", "PCommonsDN_TRANSFAC", "PCommonsDN_PhosphoSite",
"PCommonsDN_CTD"), network.customised = NULL, distance.max = 2e+05,
seed.genes = T, subnet.significance = 5e-05, subnet.size = NULL,
verbose = T,
RData.location =
"https://github.com/hfang-bristol/RDataCentre/blob/master/XGR/1.0.0")
}
\arguments{
\item{data}{a named input vector containing the sinificance level for
nodes (dbSNP). For this named vector, the element names are dbSNP, the
element values for the significance level (measured as p-value or fdr).
Alternatively, it can be a matrix or data frame with two columns: 1st
column for dbSNP, 2nd column for the significance level}

\item{include.LD}{additional SNPs in LD with Lead SNPs are also
included. By default, it is 'NA' to disable this option. Otherwise, LD
SNPs will be included based on one or more of 26 populations and 5
super populations from 1000 Genomics Project data (phase 3). The
population can be one of 5 super populations ("AFR", "AMR", "EAS",
"EUR", "SAS"), or one of 26 populations ("ACB", "ASW", "BEB", "CDX",
"CEU", "CHB", "CHS", "CLM", "ESN", "FIN", "GBR", "GIH", "GWD", "IBS",
"ITU", "JPT", "KHV", "LWK", "MSL", "MXL", "PEL", "PJL", "PUR", "STU",
"TSI", "YRI"). Explanations for population code can be found at
\url{http://www.1000genomes.org/faq/which-populations-are-part-your-study}}

\item{LD.r2}{the LD r2 value. By default, it is 0.8, meaning that SNPs
in LD (r2>=0.8) with input SNPs will be considered as LD SNPs. It can
be any value from 0.8 to 1}

\item{network}{the built-in network. Currently two sources of network
information are supported: the STRING database (version 10) and the
Pathways Commons database (version 7). STRING is a meta-integration of
undirect interactions from the functional aspect, while Pathways
Commons mainly contains both undirect and direct interactions from the
physical/pathway aspect. Both have scores to control the confidence of
interactions. Therefore, the user can choose the different quality of
the interactions. In STRING, "STRING_highest" indicates interactions
with highest confidence (confidence scores>=900), "STRING_high" for
interactions with high confidence (confidence scores>=700), and
"STRING_medium" for interactions with medium confidence (confidence
scores>=400). For undirect/physical interactions from Pathways Commons,
"PCommonsUN_high" indicates undirect interactions with high confidence
(supported with the PubMed references plus at least 2 different
sources), "PCommonsUN_medium" for undirect interactions with medium
confidence (supported with the PubMed references). For direct
(pathway-merged) interactions from Pathways Commons, "PCommonsDN_high"
indicates direct interactions with high confidence (supported with the
PubMed references plus at least 2 different sources), and
"PCommonsUN_medium" for direct interactions with medium confidence
(supported with the PubMed references). In addtion to pooled version of
pathways from all data sources, the user can also choose the
pathway-merged network from individual sources, that is,
"PCommonsDN_Reactome" for those from Reactome, "PCommonsDN_KEGG" for
those from KEGG, "PCommonsDN_HumanCyc" for those from HumanCyc,
"PCommonsDN_PID" for those froom PID, "PCommonsDN_PANTHER" for those
from PANTHER, "PCommonsDN_ReconX" for those from ReconX,
"PCommonsDN_TRANSFAC" for those from TRANSFAC, "PCommonsDN_PhosphoSite"
for those from PhosphoSite, and "PCommonsDN_CTD" for those from CTD}

\item{network.customised}{an object of class "igraph". By default, it
is NULL. It is designed to allow the user analysing their customised
network data that are not listed in the above argument 'network'. This
customisation (if provided) has the high priority over built-in
network}

\item{distance.max}{the maximum distance between genes and SNPs. Only
those genes no far way from this distance will be considered as seed
genes. This parameter will influence the distance-component weights
calculated for nearby SNPs per gene}

\item{seed.genes}{logical to indicate whether the identified network is
restricted to seed genes (ie nearby genes that are located within
defined distance window centred on lead or LD SNPs). By default, it
sets to true}

\item{subnet.significance}{the given significance threshold. By
default, it is set to NULL, meaning there is no constraint on
nodes/genes. If given, those nodes/genes with p-values below this are
considered significant and thus scored positively. Instead, those
p-values above this given significance threshold are considered
insigificant and thus scored negatively}

\item{subnet.size}{the desired number of nodes constrained to the
resulting subnet. It is not nulll, a wide range of significance
thresholds will be scanned to find the optimal significance threshold
leading to the desired number of nodes in the resulting subnet.
Notably, the given significance threshold will be overwritten by this
option}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to true for display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
a subgraph with a maximum score, an object of class "igraph"
}
\description{
\code{xSubneterSNPs} is supposed to identify maximum-scoring gene
subnetwork from an input graph with the node information on the
significance (measured as p-values or fdr). To do so, it defines seed
genes and their weights that take into account the distance to and the
significance of input SNPs. It returns an object of class "igraph".
}
\note{
The algorithm identifying a gene subnetwork that is likely modulated by
input SNPs and/or their LD SNPs includes two major steps. The first
step is to define and score nearby genes that are located within
distance window of input and/or LD SNPs. The second step is to use
\code{\link{xSubneterGenes}} for identifying a maximum-scoring gene
subnetwork that contains as many highly scored genes as possible but a
few lowly scored genes as linkers.
}
\examples{
\dontrun{
# Load the library
library(XGR)
library(igraph)
library(dnet)
library(GenomicRanges)

# a) provide the seed SNPs with the weight info
## load ImmunoBase
ImmunoBase <- xRDataLoader(RData.customised='ImmunoBase')
## get lead SNPs reported in AS GWAS and their significance info (p-values)
gr <- ImmunoBase$AS$variant
seeds.snps <- as.matrix(mcols(gr)[,c(1,3)])

# b) perform network analysis
# b1) find maximum-scoring subnet based on the given significance threshold
subnet <- xSubneterSNPs(data=seeds.snps, network="STRING_high",
seed.genes=F, subnet.significance=0.01)
# b2) find maximum-scoring subnet with the desired node number=50
subnet <- xSubneterSNPs(data=data, network="STRING_high",
subnet.size=50)

# c) save subnet results to the files called 'subnet_edges.txt' and 'subnet_nodes.txt'
output <- igraph::get.data.frame(subnet, what="edges")
utils::write.table(output, file="subnet_edges.txt", sep="\\t",
row.names=FALSE)
output <- igraph::get.data.frame(subnet, what="vertices")
utils::write.table(output, file="subnet_nodes.txt", sep="\\t",
row.names=FALSE)

# d) visualise the identified subnet
## do visualisation with nodes colored according to the significance (you provide)
xVisNet(g=subnet, pattern=-log10(as.numeric(V(subnet)$significance)),
vertex.shape="sphere", colormap="wyr")
## do visualisation with nodes colored according to transformed scores
xVisNet(g=subnet, pattern=V(subnet)$score, vertex.shape="sphere")

# e) visualise the identified subnet as a circos plot
library(RCircos)
xCircos(g=subnet, entity="Gene")
}
}
\seealso{
\code{\link{xSubneterGenes}}
}

