#'  Dynamic Spillover Index
#'
#' 
#' @description 
#' Estimates the dynamic spillover index given a moving window as described in Diebold and Yilmaz (2012). We recommend switching to \code{dynamic.spillover}.
#' 
#' @param data Object of class \sQuote{\code{zoo}}.
#' @param width An integer specifying the window width which is aligned to the original sample.
#' @param n.ahead An integer indicating the how many steps ahead the spillover should be forecasted.
#' @param index A character string indicating whether the orthogonalized or the generalized index is computed.
#' @param ortho.type A character string indicating the type of orthogonalized index is required. \code{"partial"} takes a random sample out of all the possible combinations generated for the Choleski decomposition, while \code{"total"} uses all the combinations, therefore it takes more time to finish.
#' @param ... Further arguments to be passed to \code{VAR} function from vars package.
#' 

#' @return A \code{zoo} object holding all the net spillover index estimations.

#' 
#' @references
#' Diebold, F. X. & Yilmaz, K.(2012). \emph{Better to Give than to Receive: Predictive Directional Measurement of Volatility Spillovers}. International Journal of Forecasting.
#' 
#' @author Jilber Urbina
#' @export
#' @importFrom fastSOM sot_avg_est
#' @importFrom fastSOM sot_avg_exact
#' @importFrom zoo rollapply
#' 
#' @examples
#' \donttest{
#' data(dy2012) 
#' G_net <- roll.net(as.zoo(dy2012[1:300,c(2,3,4)]), width = 200, index="generalized")
#' 
#' data(dy2012)
#' # orthogonalized rolling net spillover index, based on a VAR(2)
#' O_net_dy2012 <- roll.net(as.zoo(dy2012[,-1]), width = 200) 
#' # Generalized rolling net spillover index, based on a VAR(2)
#' G_net_dy2012 <- roll.net(as.zoo(dy2012[,-1]), width = 200, index="generalized")
#' }




roll.net <- function (data, width,  n.ahead = 10, index=c("orthogonalized", "generalized"), 
                            ortho.type=c("partial", "total"), ...) 
{
  #.Deprecated("dynamic.spillover")
  if (!inherits(data, "zoo")) {
    stop("\nPlease provide an object of class 'zoo', generated by 'as.zoo'.\n")
  }
  
  n.ahead <- as.integer(n.ahead)
  
  roll.net   <- switch(match.arg(index), 
                       orthogonalized = {
                         switch(match.arg(ortho.type), 
                                partial = {
                                  rollapply(data, width = width,
                                            FUN = function(z) {
                                              y <- VAR(z, ...)
                                              Y <- fastSOM::sot_avg_est(summary(y)$covres , Phi(y, n.ahead - 1))$Average
                                              colSums(Y-diag(diag(Y))) - rowSums(Y-diag(diag(Y)))
                                            },
                                            by.column = FALSE, align = "right") 
                                }, 
                                total = {
                                  rollapply(data, width = width,
                                            FUN = function(z) {
                                              y <- VAR(z, ...)
                                              Y <- fastSOM::sot_avg_exact(summary(y)$covres , Phi(y, n.ahead - 1))$Average
                                              colSums(Y-diag(diag(Y))) - rowSums(Y-diag(diag(Y)))
                                            },
                                            by.column = FALSE, align = "right") 
                                } )
                         
                         
                       }, 
                       generalized ={
                         K <- ncol(data)
                         rollapply(data, width = width,
                                   FUN = function(z) {
                                     Y <-  G.spillover(VAR(z,...), n.ahead=n.ahead)[1:K,1:K]
                                     colSums(Y-diag(diag(Y))) - rowSums(Y-diag(diag(Y)))
                                   },
                                   by.column = FALSE, align = "right") 
                       }  )
  
  
  
  return(roll.net)
  
}



