\name{map.latent}
\alias{map.latent}
\title{Two dimensional map from a Bayesian hierarchical model}
\description{
  This function plots 2D maps from a Markov chain.
}
\usage{
map.latent(fitted, x, y, covariates = NULL, param = "quant", ret.per =
100, col = terrain.colors(64), plot.contour = TRUE, fun = mean, level =
0.95, show.data = TRUE, control = list(nlines = 500), ...)
}
\arguments{
  \item{fitted}{An object of class "latent". Typically this will be the
    output of \code{\link{latent}}.}
  \item{x,y}{Numeric vector specifying the coordinates of the grid points.}
  \item{covariates}{An array specifying the covariates at each grid
    point defined by \code{x} and \code{y}. If \code{NULL}, no covariate
    is needed.}
  \item{param}{A character string. Must be one of "loc", "scale",
    "shape" or "quant" for a map of the location, scale, shape
    parameters or for a map of a specified quantile.}
  \item{ret.per}{A numeric giving the return period for which the
    quantile map is plotted. It is only required if \code{param =
      "quant"}.}
  \item{col}{A list of colors such as that generated by 'rainbow',
    'heat.colors', 'topo.colors', 'terrain.colors' or similar
    functions.}
  \item{plot.contour}{Logical. If \code{TRUE} (default), contour lines
    are added to the plot.}
  \item{fun}{A character string specifying the function to be used to
    get posterior point estimates. The default is to take posterior
    means.}
  \item{level}{A numeric specifying the significance level for the
    pointwise credible intervals.}
  \item{show.data}{Logical. Should the locations where have observed the
    process have to be plotted?}
  \item{control}{A list with named components specifying options to be
    passed to \code{\link{rgp}}. Typically one might want specify
    \code{nlines} to reduce the computational demand.}
  \item{\dots}{Several arguments to be passed to the \code{\link{image}}
    function.}
}
\value{
  A plot and a invisible list containing all the data required to do the
  plot.
}

\author{Mathieu Ribatet}
\seealso{\code{\link{condrgp}}, \code{\link{map}}}
\examples{
## Generate realizations from the model
n.site <- 30
n.obs <- 50
coord <- cbind(lon = runif(n.site, -10, 10), lat = runif(n.site, -10 , 10))

gp.loc <- rgp(1, coord, "powexp", sill = 4, range = 20, smooth = 1)
gp.scale <- rgp(1, coord, "powexp", sill = 0.4, range = 5, smooth = 1)
gp.shape <- rgp(1, coord, "powexp", sill = 0.01, range = 10, smooth = 1)

locs <- 26 + 0.5 * coord[,"lon"] + gp.loc
scales <- 10 + 0.2 * coord[,"lat"] + gp.scale
shapes <- 0.15 + gp.shape

data <- matrix(NA, n.obs, n.site)
for (i in 1:n.site)
  data[,i] <- rgev(n.obs, locs[i], scales[i], shapes[i])

loc.form <- y ~ lon
scale.form <- y ~ lat
shape.form <- y ~ 1

hyper <- list()
hyper$sills <- list(loc = c(1,8), scale = c(1,1), shape = c(1,0.02))
hyper$ranges <- list(loc = c(2,20), scale = c(1,5), shape = c(1, 10))
hyper$smooths <- list(loc = c(1,1/3), scale = c(1,1/3), shape = c(1, 1/3))
hyper$betaMeans <- list(loc = rep(0, 2), scale = c(9, 0), shape = 0)
hyper$betaIcov <- list(loc = solve(diag(c(400, 100))),
                       scale = solve(diag(c(400, 100))),
                       shape = solve(diag(c(10), 1, 1)))

## We will use an exponential covariance function so the jump sizes for
## the shape parameter of the covariance function are null.
prop <- list(gev = c(2.5, 1.5, 0.2), ranges = c(0.7, 0.75, 0.9), smooths = c(0,0,0))
start <- list(sills = c(4, .36, 0.009), ranges = c(24, 17, 16), smooths
              = c(1, 1, 1),  beta = list(loc = c(26, 0.5), scale = c(10, 0.2),
                               shape = c(0.15)))

## Generate a Markov chain
mc <- latent(data, coord, loc.form = loc.form, scale.form = scale.form,
             shape.form = shape.form, hyper = hyper, prop = prop, start = start,
             n = 100)

x.grid <- y.grid <- seq(-10, 10, length = 50)
map.latent(mc, x.grid, y.grid, param = "shape")
}
\keyword{hplot}

