test_that("simulation runs with simInit and spades", {
  library(igraph); on.exit(detach("package:igraph"), add = TRUE)

  set.seed(42)

  times <- list(start = 0.0, end = 10, timeunit = "year")
  params <- list(
    .globals = list(burnStats = "npixelsburned", stackName = "landscape"),
    randomLandscapes = list(.plotInitialTime = NA, .plotInterval = NA),
    caribouMovement = list(.plotInitialTime = NA, .plotInterval = NA, torus = TRUE),
    fireSpread = list(.plotInitialTime = NA, .plotInterval = NA)
  )
  modules <- list("randomLandscapes", "caribouMovement", "fireSpread")
  paths <- list(modulePath = system.file("sampleModules", package = "SpaDES.core"))

  mySim <- simInit(times, params, modules, objects = list(), paths) %>% spades()

  # simtime
  expect_equivalent(time(mySim), 10.0)
  expect_equivalent(start(mySim), 0.0)
  expect_equivalent(end(mySim), 10.0)

  # sim results ## NOTE upcoming version of RandomFields completely changes the values!!!
  if (packageVersion("RandomFields") >= "3.1.20") {
    burnedLast <- c(1435, 1044, 1531, 844, 1093, 1379, 2026, 1181, 469, 1381)

    pos_x <- c(33.2872398995189, -17.3422621469692, 8.75313234185035,
               24.0077471712237, -40.9511657271224, 4.67136848352605, 1.63754197323153,
               47.0523572578574, -33.249115084004, 48.0873109754402, -7.77153078573273,
               -48.9946323868193, 17.9728675881409, 37.6335473250708, 5.39323021033027,
               -47.4864772203972, 11.6040269926992, -0.865950034505744, -36.9293704235718,
               -21.531106341633, 40.9938873771294, -42.5429308603906, -25.9181746115019,
               -26.2432233488535, 14.0087500069471, 25.8855766724122, 46.9359618222115,
               -19.8016223717072, -49.8222051620967, 30.2757492095112, -7.28689325262428,
               -4.50039315271449, -10.547261947459, -11.5071818522213, 4.48358610873886,
               25.6944051540438, -41.1416895584928, 9.48882488424441, -18.4614444331997,
               34.9689888922441, 0.840919977329037, -41.0611183006543, 0.795379844279644,
               -24.1159885913404, 1.42940853742768, 5.07830438400629, -7.7835339336879,
               -49.9176129132604, 38.0547415109303, -23.617696164462, 27.598907121919,
               4.52221827777863, 6.58707710429276, 48.534921591503, -11.7141259923688,
               47.7952502894699, -40.0630795353281, -20.6192855655735, 3.87439431752425,
               -14.3781330735223, -15.2560485462677, -31.5638425447865, 48.8032870922823,
               5.81793650533037, 41.8727853065613, -30.0912034365545, 46.5808437960944,
               15.7969872175151, 43.2982483637298, 0.722432416500425, -21.2889097741673,
               -24.092352778194, 42.8997225069988, 38.2804348365648, -40.6515240164646,
               25.4967712722791, 31.9804489132599, -40.6608411861367, 10.4166482355862,
               -15.1510045022174, 20.0076631157052, 43.0570386794077, -24.9791258717695,
               -29.8709993334512, 45.3060116671438, 28.3291482755146, -43.5158165322117,
               39.4175768516032, 20.3064395385905, -37.3700933983955, 32.7448779315204,
               33.6911235142027, -25.8218432960084, -6.64749387632853, -37.2734122354639,
               -24.701424556876, -45.1535543851411, 5.4222549212579, -21.262059193088,
               7.08587571124283)

    pos_y <- c(-36.7915302068531, 19.4049257142821, 8.99246407697373,
               11.1972141131503, -28.9042034294489, 21.6276219077891, -31.0277712593001,
               -41.7658238188155, -25.3298671454348, -11.8963315269705, -41.3248994981119,
               -17.5279823276148, -7.52501323292238, -29.5785339759454, 2.02411986858414,
               -26.9279490346496, 37.0220847585067, 33.01216290486, -36.7155731663296,
               -20.3595762313143, 20.2377052688342, -48.731190870263, -34.0058211864778,
               -12.7019908770908, 2.85640042972667, -6.81740166205414, -22.8625105848804,
               -20.8366231499668, 35.7474349085467, 11.8197161345573, -2.9113717513449,
               -17.5537959621871, 5.53538898703373, 24.2233244841148, -3.38930417352202,
               -45.6155550309963, 6.48087493076159, -46.1985693927484, -2.96529308770018,
               21.1228741105041, -31.5132944934996, -11.9282793587603, -14.3123527346389,
               -9.27831516836018, -38.7022504195187, -5.89356892562855, 42.92261409509,
               -34.626156901399, -0.830322963815291, 22.4308200079755, -32.8706273407653,
               -36.3802314149427, 4.50055526477191, -42.26773250131, -14.6343299094686,
               -45.7342538278528, -4.9852604991062, 18.2115594208981, -27.2099645387977,
               -29.9166461890478, 8.1878785078301, -20.2503321669608, -47.3353166806515,
               -22.4684753660724, -4.28818807965439, -29.6287278762122, -35.8334444728408,
               -5.38698453074952, 32.3835464252082, -7.82200530744319, 2.52995097890168,
               -45.3450571270551, 45.5202912871924, 29.8891155371416, -46.7513360594827,
               34.8309029291136, -23.3302083002779, 29.2185922344093, 26.6638390625404,
               -11.0415248404228, 22.8039989094315, 36.0049451333158, 7.31257801889006,
               -48.8194164055657, -36.9492164182348, -10.6970705800586, -48.1786489911048,
               -19.3699919165668, 11.3725621864949, -18.7864877048488, -42.1370407807376,
               23.5682143369233, -30.4867970470337, -30.7283217366377, -11.6149434287642,
               22.6263984629265, 9.66250955670057, -22.3417192277185, -6.44269250619922,
               -14.501302544863)
  } else {
    burnedLast <- c(1680, 1485, 607, 1079, 1041, 605, 871, 1097, 495, 1253)

    pos_x <- c(
      -36.9964594878183, 2.32458656754471, 30.12442850383, -18.6640229184998,
      -33.8885207519506, -29.5427930908878, -21.0095214973779, -6.42587067738999,
      -5.07663946918247, 1.51100547592112, 40.1425992497513, -40.0937640741396,
      8.4223912961982, -28.2165542018589, -23.0437975158775, 18.294604333076,
      -16.7308043319382, -8.47854059870558, 39.0343113708618, -39.3309621495378,
      -3.85775142502581, 49.0817194947727, -17.9722065247698, 4.55173210595375,
      -41.0519469414064, 34.8930158926542, -32.589908783801, -40.5387963961559,
      34.2194237750273, -33.5059996554471, 41.7214409962314, -33.4975937591762,
      -16.7534188285129, -3.15227992913145, 23.6589792431533, 28.0041852235432,
      43.3963008307616, 15.6253468583625, -24.338203383424, 38.3698431106254,
      -8.95270144561675, 33.179905218285, -32.412733860245, 35.8000608503506,
      32.1821466587538, -5.44818140354121, 34.3391843159336, -1.11939974258066,
      -29.9411578108715, -8.4957082608383, 38.4884378359319, -44.6014019956853,
      9.52576850503965, -29.621391887535, -3.42226265233396, -48.0427755267367,
      -48.0453925992576, 31.3356691703378, 39.4370108644243, -23.0450600175721,
      -29.9676867807934, -14.984320542406, 13.0752058955144, 11.8531604473534,
      -24.9697318775997, -25.3771063203708, 46.0558653502021, 27.1583913421414,
      47.0730137699115, -43.2589915711839, 34.287440364578, -16.462781379318,
      13.8025952614028, 7.98145114328579, 12.6617467700004, 8.6529212125842,
      42.9266926532228, -29.6405295858366, 35.0331975539472, 22.782823127803,
      -33.6543726032234, -9.20996524772551, 6.86344094726754, -42.0192917407262,
      41.097913592971, 9.11335978963641, 35.7427369785196, 1.39434498353882,
      25.6514918771879, -25.7249927901813, 46.5803504144238, -46.9702484362805,
      30.3177776221904, 11.1446124689632, 46.5656200410542, 42.6072217133873,
      -26.5944387764709, 30.3706355061775, -39.2953280003378, -6.61414387806415
    )

    pos_y <- c(
      11.2259177001642, -48.2749285467045, -19.2693150119305, -30.1955629122566,
      -5.16274098908813, 19.7263653977621, -18.3682115225133, 11.738367415035,
      38.9967732966357, 2.71795967614925, -44.7225009655404, -32.6275290676451,
      -14.7686749801842, 25.2266058813999, 43.367709252435, 40.625866000042,
      22.0714768248821, -24.1975134747756, -24.2623383782744, -12.5286480447744,
      48.122230154392, -44.8308278925359, 37.0642433839201, -15.6006743461834,
      -28.265637676497, 8.5281444890346, 37.3382373812721, 30.2225238433243,
      -26.9394661142134, 29.4716759594875, -43.5670094907836, -47.8084247612384,
      -48.1749534451242, -45.1487680482165, -30.8568223487393, -39.0221779181183,
      -15.806612632621, 9.37725216531152, -40.2347020337625, 19.3362330671687,
      10.0653991235251, -24.599586469326, -35.8913143320209, 35.6277170387067,
      35.5608652339012, -36.7887704649442, -5.75447090913782, 1.01447582952702,
      12.9836719196659, -18.5773516046411, -46.6496786531966, -22.4229092556123,
      35.9359685267436, 45.2300691122058, 25.6432668008571, 41.3834115900633,
      -13.4068360644368, -26.1410488438946, -47.4071373813623, -32.2387795954974,
      -35.7842792759393, -7.70426791095162, 36.0018667789867, 30.1562830869532,
      41.6895906080711, 47.8520336608373, 3.57737369062482, -39.6190773865091,
      5.37402201117737, -37.3107079155598, -34.1742440396536, -19.2538176414933,
      -48.6308373737092, -26.392445100473, -41.039612082825, -46.6119580572095,
      -16.844154891972, -26.0741751607543, -31.4374762519959, -6.44047912640615,
      44.8723281786276, -43.6527354527293, 37.8112535875202, -22.5498526193275,
      25.9669403382768, 1.16676526552681, -30.6497237706147, 32.5573215765397,
      15.4977777956788, 20.7609180017956, 31.0985715416653, -34.7622261504206,
      18.6274139594293, 43.0727309557879, -18.0134694585456, 39.5790564364162,
      12.5097325118235, -31.4932896470479, 37.7529892755605, -30.960625150814
    )
  }

  expect_equal(mySim$npixelsburned, burnedLast)
  expect_equivalent(mySim$caribou$x, pos_x)
  expect_equivalent(mySim$caribou$y, pos_y)
})

test_that("spades calls with different signatures don't work", {
  library(igraph)
  library(reproducible)

  tmpdir <- file.path(tempdir(), "test_Plot1") %>% checkPath(create = TRUE)
  cwd <- getwd()
  setwd(tmpdir)

  on.exit({
    detach("package:reproducible")
    detach("package:igraph")
    setwd(cwd)
    unlink(tmpdir, recursive = TRUE)
  }, add = TRUE)

  a <- simInit()
  a1 <- Copy(a)
  expect_silent(spades(a))
  expect_output(spades(a, debug = TRUE), "eventTime")
  expect_silent(spades(a, .plotInitialTime = NA))
  expect_silent(spades(a, .saveInitialTime = NA))
  expect_output(spades(a, debug = TRUE, .plotInitialTime = NA), "eventTime")
  expect_output(spades(a, debug = TRUE, .saveInitialTime = NA), "eventTime")
  expect_equivalent(capture_output(spades(a, debug = "current", .plotInitialTime = NA)),
                    capture_output(spades(a, debug = TRUE, .plotInitialTime = NA)))

  expect_output(spades(a, debug = c("current", "events"), .plotInitialTime = NA),
                "This is the current event")
  expect_output(spades(a, debug = c("current", "events"), .plotInitialTime = NA),
                "moduleName")
  expect_output(spades(a, debug = "simList", .plotInitialTime = NA),
                "Completed Events")

  if (interactive()) {
    expect_output(spades(a, progress = "text", debug = TRUE), "10%")
    expect_output(spades(a, progress = "text", debug = TRUE), "20%")
    expect_output(spades(a, progress = "text"), "..........| 100%")
  }
  expect_silent(spades(a, progress = FALSE))
  expect_silent(spades(a, progress = "rr"))

  paths(a)$cachePath <- file.path(tempdir(), "cache") %>% checkPath(create = TRUE)
  a <- Copy(a1)
  expect_output(spades(a, cache = TRUE, debug = TRUE, notOlderThan = Sys.time()), "eventTime")
  expect_true(all(dir(paths(a)$cachePath) == c("backpack.db", "gallery")))
  file.remove(dir(paths(a)$cachePath, full.names = TRUE, recursive = TRUE))

  # test for system time ... in this case, the first time through loop is slow
  #   because of writing cache to disk, not because of spades being slow.
  #   SimList is empty.

  set.seed(42)

  times <- list(start = 0.0, end = 0, timeunit = "year")
  params <- list(
    .globals = list(burnStats = "npixelsburned", stackName = "landscape"),
    randomLandscapes = list(nx = 20, ny = 20)
  )
  modules <- list("randomLandscapes", "fireSpread")
  paths <- list(modulePath = system.file("sampleModules", package = "SpaDES.core"))

  for (i in 1:2) {
    a <- simInit(times, params, modules, paths = paths)
    paths(a)$cachePath <- file.path(tempdir(), "cache") %>% checkPath(create = TRUE)
    assign(paste0("st", i), system.time(spades(a, cache = TRUE, .plotInitialTime = NA)))
  }
  expect_gt(st1[1], st2[1])
  file.remove(dir(paths(a)$cachePath, full.names = TRUE, recursive = TRUE))
})

test_that("simInit with R subfolder scripts", {
  library(igraph)
  library(reproducible)

  tmpdir <- file.path(tempdir(), "test_timeunits") %>% checkPath(create = TRUE)
  cwd <- getwd()
  setwd(tmpdir)

  on.exit({
    detach("package:reproducible")
    detach("package:igraph")
    setwd(cwd)
    unlink(tmpdir, recursive = TRUE)
  }, add = TRUE)

  newModule("child1", ".", open = FALSE)
  cat(file = file.path("child1", "R", "script.R"),
      "a <- function(r) {
          r + 1
      }", sep = "\n")
  mySim <- simInit(modules = "child1", paths = list(modulePath = tmpdir))
  expect_true(sum(grepl(unlist(lapply(ls(mySim@.envir, all.names = TRUE), function(x) {
    if (is.environment(mySim@.envir[[x]])) ls(envir = mySim@.envir[[x]], all.names = TRUE)
  })), pattern = "^a$")) == 1)
  expect_true(mySim@.envir$._child1$a(2) == 3)
})

test_that("simulation runs with simInit with duplicate modules named", {
  library(igraph); on.exit(detach("package:igraph"), add = TRUE)

  set.seed(42)

  times <- list(start = 0.0, end = 10, timeunit = "year")
  params <- list(
    randomLandscapes = list(.plotInitialTime = NA, .plotInterval = NA),
    caribouMovement = list(.plotInitialTime = NA, .plotInterval = NA, torus = TRUE)
  )
  modules <- list("randomLandscapes", "randomLandscapes", "caribouMovement")
  paths <- list(modulePath = system.file("sampleModules", package = "SpaDES.core"))

  expect_true(grepl(capture_messages(
    mySim <- simInit(times, params, modules, objects = list(), paths)
  ), pattern = "Duplicate module"))
  expect_true(length(modules(mySim)) != length(modules))
  expect_true(length(modules(mySim)) == length(unique(modules)))
})


test_that("simulation runs with simInit with duplicate modules named", {
  skip("benchmarking DES")

  library(dplyr)

  tmpdir <- file.path(tempdir(), "testBenchmarking")
  checkPath(tmpdir, create = TRUE)
  setwd(tmpdir)
  on.exit(unlink(tmpdir, force = TRUE, recursive = TRUE))
  newModule("test", tmpdir)

  sim <- simInit()

  cat(file = file.path(tmpdir, "test", "test.R"),'
  defineModule(sim, list(
    name = "test",
    description = "insert module description here",
    keywords = c("insert key words here"),
    authors = person(c("Eliot", "J", "B"), "McIntire", email = "eliot.mcintire@canada.ca", role = c("aut", "cre")),
    childModules = character(0),
    version = list(SpaDES.core = "0.1.0", test = "0.0.1"),
    spatialExtent = raster::extent(rep(NA_real_, 4)),
    timeframe = as.POSIXlt(c(NA, NA)),
    timeunit = "year",
    citation = list("citation.bib"),
    documentation = list("README.txt", "test.Rmd"),
    reqdPkgs = list(),
    parameters = rbind(
    ),
    inputObjects = bind_rows(
    ),
    outputObjects = bind_rows(
    )
  ))

  doEvent.test = function(sim, eventTime, eventType, debug = FALSE) {
    switch(
      eventType,
      init = {
        sim <- scheduleEvent(sim, time(sim)+1, "test", "event1")
      },
      event1 = {
        sim <- scheduleEvent(sim, time(sim)+1, "test", "event1")
      })
    return(invisible(sim))
  }
  ', fill = TRUE)

  library(igraph)

  moduleDir <- file.path(tmpdir)
  inputDir <- file.path(moduleDir, "inputs") %>% reproducible::checkPath(create = TRUE)
  outputDir <- file.path(moduleDir, "outputs")
  cacheDir <- file.path(outputDir, "cache")
  times <- list(start = 0, end = 5000)
  parameters <- list(
  )
  modules <- list("test")
  objects <- list()
  paths <- list(
    cachePath = cacheDir,
    modulePath = moduleDir,
    inputPath = inputDir,
    outputPath = outputDir
  )

  mySim <- simInit(times = times, params = parameters, modules = modules,
                   objects = objects, paths = paths)

  # was 10.2 seconds -- currently 4.2 seconds or so
  #system.time({spades(mySim, debug = FALSE)})
  microbenchmark::microbenchmark(times = 10, {spades(mySim, debug = FALSE)})
  #profvis::profvis({spades(mySim, debug = FALSE)})
})
