% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ABCSMC.R
\name{ABCSMC}
\alias{ABCSMC}
\alias{ABCSMC.ABCSMC}
\alias{ABCSMC.default}
\title{Runs ABC-SMC algorithm}
\usage{
ABCSMC(x, ...)

\method{ABCSMC}{ABCSMC}(
  x,
  tols = NULL,
  ptols = NULL,
  mintols = NULL,
  ngen = 1,
  parallel = FALSE,
  mc.cores = NA,
  ...
)

\method{ABCSMC}{default}(
  x,
  priors,
  func,
  u,
  tols = NULL,
  ptols = NULL,
  mintols = NULL,
  ngen = 1,
  npart = 100,
  parallel = FALSE,
  mc.cores = NA,
  ...
)
}
\arguments{
\item{x}{An \code{ABCSMC} object or a named vector with entries
containing the observed summary statistics to match to. Names must match to `tols`.}

\item{...}{Further arguments to pass to \code{func}. (Not used if extending runs.)}

\item{tols}{A \code{vector} or \code{matrix} of tolerances, with the number of rows defining
the number of generations required, and columns defining the summary statistics
to match to. If a \code{vector}, then the length determines the summary statistics.
The columns/entries must match to those in `x`.}

\item{ptols}{The proportion of simulated outcomes at each generation to use to derive adaptive 
tolerances.}

\item{mintols}{A vector of minimum tolerance levels.}

\item{ngen}{The number of generations of ABC-SMC to run.}

\item{parallel}{A \code{logical} determining whether to use parallel processing or not.}

\item{mc.cores}{Number of cores to use if using parallel processing.}

\item{priors}{A \code{data.frame} containing columns \code{parnames}, \code{dist}, \code{p1} and 
\code{p2}, with number of rows equal to the number of parameters. The column
\code{parname} simply gives names to each parameter for plotting and summarising.
Each entry in the \code{dist} column must contain one of \code{c("unif", "norm", "gamma")}, 
and the corresponding \code{p1} and \code{p2} entries relate to the hyperparameters 
(lower and upper bounds in the uniform case; mean and standard deviation in the 
normal case; and shape and rate in the gamma case).}

\item{func}{Function that runs the simulator and checks whether the simulation matches the data. 
The first four arguments must be \code{pars}, \code{data}, \code{tols} and 
\code{u}. If the simulations do not match the data then the function must 
return an \code{NA}, else it must returns a \code{vector} of simulated summary measures. 
In this latter case the output from the function must be a vector with length equal to 
\code{ncol(data)} and with entries in the same order as the columns of \code{data}.}

\item{u}{A named vector of initial states.}

\item{npart}{An integer specifying the number of particles.}
}
\value{
An \code{ABCSMC} object, essentially a \code{list} containing:
\itemize{
\item{\code{pars}:}{ a \code{list} of \code{matrix} objects containing the accepted
                     particles. Each element of the list corresponds to a generation 
                     of ABC-SMC, with each matrix being of dimension 
                     \code{npart} x \code{npars};}
\item{\code{output}:}{ a \code{list} of \code{matrix} objects containing the simulated
                     summary statistics. Each element of the list corresponds to a
                     generation of ABC-SMC, with each matrix being of dimension 
                     \code{npart} x \code{ncol(data)};}
\item{\code{weights}:}{ a \code{list} of \code{vector} objects containing the particle
                     weights. Each element of the list corresponds to a
                     generation of ABC-SMC, with each vector being of length
                     \code{npart};}
\item{\code{ESS}:}{ a \code{list} of effective sample sizes. Each element of the list 
                     corresponds to a generation of ABC-SMC, with each vector being of 
                     length \code{npart};}
\item{\code{accrate}:}{ a \code{vector} of length \code{nrow(tols)} containing the
                     acceptance rates for each generation of ABC;}
\item{\code{tols}:}{ a copy of the \code{tols} input;}
\item{\code{ptols}:}{ a copy of the \code{ptols} input;}
\item{\code{mintols}:}{ a copy of the \code{mintols} input;}
\item{\code{priors}:}{ a copy of the \code{priors} input;}
\item{\code{data}:}{ a copy of the \code{data} input;}
\item{\code{func}:}{ a copy of the \code{func} input;}
\item{\code{u}}{ a copy of the \code{u} input;}
\item{\code{addargs}:}{ a copy of the \code{...} inputs.}
}
}
\description{
Runs the Approximate Bayesian Computation Sequential Monte Carlo 
    (ABC-SMC) algorithm of Toni et al. (2009) for fitting 
    infectious disease models to time series count data.
}
\details{
Samples initial particles from the specified prior distributions and 
    then runs a series of generations of ABC-SMC. The generations can either be 
    specified with a set of fixed tolerances, or by setting the tolerances at 
    each new generation as a quantile of the tolerances of the accepted particles 
    at the previous generation. Uses bisection method as detailed in McKinley et al. (2018).
    Passing an \code{ABCSMC} object into the \code{ABCSMC()} function acts as a 
    continuation method, allowing further generations to be run.
}
\examples{
\donttest{
## set up SIR simulationmodel
transitions <- c(
    "S -> beta * S * I -> I", 
    "I -> gamma * I -> R"
)
compartments <- c("S", "I", "R")
pars <- c("beta", "gamma")
model <- mparseRcpp(
    transitions = transitions, 
    compartments = compartments,
    pars = pars
)
model <- compileRcpp(model)

## generate function to run simulators
## and return summary statistics
simSIR <- function(pars, data, tols, u, model) {

    ## run model
    sims <- model(pars, 0, data[2] + tols[2], u)
    
    ## this returns a vector of the form:
    ## completed (1/0), t, S, I, R (here)
    if(sims[1] == 0) {
        ## if simulation rejected
        return(NA)
    } else {
        ## extract finaltime and finalsize
        finaltime <- sims[2]
        finalsize <- sims[5]
    }
    
    ## return vector if match, else return NA
    if(all(abs(c(finalsize, finaltime) - data) <= tols)){
        return(c(finalsize, finaltime))
    } else {
        return(NA)
    }
}

## set priors
priors <- data.frame(
    parnames = c("beta", "gamma"), 
    dist = rep("gamma", 2), 
    stringsAsFactors = FALSE
)
priors$p1 <- c(10, 10)
priors$p2 <- c(10^4, 10^2)

## define the targeted summary statistics
data <- c(
    finalsize = 30, 
    finaltime = 76
)

## set initial states (1 initial infection 
## in population of 120)
iniStates <- c(S = 119, I = 1, R = 0)

## set initial tolerances
tols <- c(
    finalsize = 50,
    finaltime = 50
)

## run 2 generations of ABC-SMC
## setting tolerance to be 50th
## percentile of the accepted 
## tolerances at each generation
post <- ABCSMC(
    x = data, 
    priors = priors, 
    func = simSIR, 
    u = iniStates, 
    tols = tols, 
    ptol = 0.2, 
    ngen = 2, 
    npart = 50,
    model = model
)
post

## run one further generation
post <- ABCSMC(post, ptols = 0.5, ngen = 1)
post
summary(post)

## plot posteriors
plot(post)

## plot outputs
plot(post, "output")
}

}
\references{
Toni T, Welch D, Strelkowa N, Ipsen A and Stumpf MP (2009) <doi:10.1098/rsif.2008.0172>

McKinley TJ, Cook AR and Deardon R (2009) <doi:10.2202/1557-4679.1171>

McKinley TJ, Vernon I, Andrianakis I, McCreesh N, Oakley JE, Nsubuga RN, Goldstein M and White RG (2018) <doi:10.1214/17-STS618>
}
\seealso{
\code{\link{print.ABCSMC}}, \code{\link{plot.ABCSMC}}, \code{\link{summary.ABCSMC}}
}
