% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sites_to_lsn.R
\name{sites_to_lsn}
\alias{sites_to_lsn}
\title{Incorporate sites into a Landscape Network}
\usage{
sites_to_lsn(
  sites,
  edges,
  snap_tolerance,
  save_local = TRUE,
  lsn_path = NULL,
  file_name = NULL,
  overwrite = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{sites}{An \code{sf} object with POINT geometry, typically
representing observed data or prediction locations.}

\item{edges}{An \code{sf} object with LINESTING geometry created
using \code{\link{lines_to_lsn}}.}

\item{snap_tolerance}{Numeric distance in map units >= 0. Sites that
are <= this distance from an edge feature will snapped to the
closest location on the edge. When the distance between the site
and all edges is > \code{snap_tolerance}, the point feature is not
snapped or included in the lsn_path.}

\item{save_local}{Logical indicating whether the outputs should be
saved to a local directory in GeoPackage format. Defaults to
\code{TRUE}.}

\item{lsn_path}{Pathname to the LSN. This is typically a directory created by
\code{\link{lines_to_lsn}}. Required if \code{save_local = TRUE}.}

\item{file_name}{Filename for output sites, which are saved to \code{lsn_path}
in GeoPackage format (must include the .gpkg extension). Required if \code{save_local = TRUE}.}

\item{overwrite}{Logical indicating whether the outputs saved to
\code{lsn_path} should overwrite existing files if they
exist. Defaults to \code{FALSE}.}

\item{verbose}{Logical indicating whether progress messages should
be printed to the console. Defaults to \code{TRUE}}
}
\value{
An \code{sf} object with POINT geometry containing the
features from \code{sites} that were found within the
\code{snap_tolerance} distance to the closest edge. In addition to
the original columns in \code{sites}, three new columns are
added: rid, ratio, and snapdist (see Details). If
\code{save_local = TRUE}, the \code{sf} object is also saved to
\code{lsn_path}.
}
\description{
Incorporates an \code{sf} object containing features
with POINT geometry into a landscape network (LSN), which is a
topological data model of streams/rivers represented as a
directional graph embedded in 2-D geographic space. Point
locations are 'snapped' to the closest edge location and new
information is generated describing the geographic and
topological location relative to other features in the LSN.
}
\details{
The \code{sites_to_lsn} function is used to incorporate
observed and prediction sites into the LSN. The output is an
\code{sf} object with POINT geometry, which contains only the
point features from \code{sites} that were found less than the
\code{snap_tolerance} distance from the closest edge
feature. When a \code{sites} point feature meets these
requirements, it is moved (i.e. snapped) to the closest location
on an edge feature. Three new columns are also added: rid, ratio,
and snapdist. The rid column contains the rid for the edge the
site has been snapped to. The second column, ratio,
represents the proportional length of the edge found between the
downstream end node for the edge and the updated/snapped site
location. The snapdist is the distance in map units that the site
was moved. If the distance between a point feature and the
closest edge is greater than or equal to the
\code{snap_tolerance}, the feature is not included in the output.

The \code{snap_tolerance} must always be \eqn{\ge 0} and must be large
enough to snap all of the point features to the edges. Using
\code{snap_tolerance = 0} is not recommended, even when the
\code{sites} features intersect the edge features. Instead, a
very small \code{snap_tolerance} value is recommended to ensure
that differences in the precision of the x and y coordinates and
the line location do not result in unsnapped point features.

Note that the \code{sites} and \code{edges}
must have the same projection.
}
\examples{
# Get temporary directory, where the example LSN will be stored
# locally.
temp_dir <- tempdir()
# Build the LSN. When working with your own data, lsn_path will be
# a local folder of your choice rather than a temporary directory.
edges<- lines_to_lsn(
   streams = MF_streams,
   lsn_path = temp_dir,
   snap_tolerance = 1,
   check_topology = FALSE,
   overwrite = TRUE,
   verbose = FALSE
)

# # Incorporate observed sites, MF_obs, into LSN
obs<- sites_to_lsn(
   sites = MF_obs,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 100,
   overwrite = TRUE,
   verbose = FALSE
)
# Notice that 3 new columns have been added to obs
names(obs)

# Incorporate prediction dataset, MF_preds, into LSN
preds<- sites_to_lsn(sites = MF_preds,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 1,
   overwrite = TRUE,
   verbose = FALSE
)

}
