% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SEMhelp.R
\name{predict.SEM}
\alias{predict.SEM}
\title{SEM-based out-of-sample prediction using layer-wise ordering}
\usage{
\method{predict}{SEM}(object, newdata, verbose = FALSE, ...)
}
\arguments{
\item{object}{An object, as that created by the function \code{SEMrun()}
with the argument \code{fit} set to \code{fit = 0} or \code{fit = 1}.}

\item{newdata}{A matrix with new data, with rows corresponding to subjects,
and columns to variables. If \code{object$fit} is a model with the group
variable (\code{fit = 1}), the first column of newdata must be the new
group binary vector (0=control, 1=case).}

\item{verbose}{A logical value. If FALSE (default), the processed graph 
will not be plotted to screen.}

\item{...}{Currently ignored.}
}
\value{
A list of 2 objects:
\enumerate{
\item "PE", vector of the prediction error equal to the Mean Squared Error
(MSE) for each out-of-bag prediction. The first value of PE is the AMSE,
where we average over all (sink and mediators) graph nodes.
\item "Yhat", the matrix of continuous predicted values of graph nodes  
(excluding source nodes) based on out-of-bag samples. 
}
}
\description{
Given the values of (observed) x-variables in a SEM,
this function may be used to predict the values of (observed) y-variables.
The predictive procedure consists of two steps: (1) construction of the
topological layer (TL) ordering of the input graph; (2) prediction of
the node y values in a layer, where the nodes included in the previous
layers act as predictors x.
}
\details{
The function first creates a layer-based structure of the
input graph. Then, a SEM-based predictive approach (Rooij et al., 2022) 
is used to produce predictions while accounting for the graph structure
organised in topological layers, j=1,...,L. In each iteration, the response
variables y are the nodes in the j layer and the predictors x are the nodes
belonging to the previous j-1 layers. 
Predictions (for y given x) are based on the (joint y and x) model-implied 
variance-covariance (Sigma) matrix and mean vector (Mu) of the fitted SEM,
and the standard expression for the conditional mean of a multivariate normal
distribution. Thus, the layer structure described in the SEM is taken into
consideration, which differs from ordinary least squares (OLS) regression.
}
\examples{

# load ALS data
ig<- alsData$graph
data<- alsData$exprs
data<- transformData(data)$data
group<- alsData$group

#...with train-test (0.5-0.5) samples
set.seed(123)
train<- sample(1:nrow(data), 0.5*nrow(data))

# SEM fitting
sem0<- SEMrun(ig, data[train,], SE="none", limit=1000)

# predictors, source+mediator; outcomes, mediator+sink

res0<- predict(sem0, newdata=data[-train,]) 
print(res0$PE)

# SEM fitting
sem1<- SEMrun(ig, data[train,], group[train], SE="none", limit=1000)

# predictors, source+mediator+group; outcomes, source+mediator+sink

res1<- predict(sem1, newdata=cbind(group,data)[-train,]) 
print(res1$PE)

\donttest{
#...with a binary outcome (1=case, 0=control)

ig1<- mapGraph(ig, type="outcome"); gplot(ig1)
outcome<- ifelse(group == 0, -1, 1); table(outcome)
data1<- cbind(outcome, data); data1[1:5,1:5]

sem10 <- SEMrun(ig1, data1[train,], SE="none", limit=1000)
res10<- predict(sem10, newdata=data1[-train,], verbose=TRUE) 

yobs<- group[-train]
yhat<- res10$Yhat[,"outcome"]
benchmark(yobs, yhat)

#...with predictors, source nodes; outcomes, sink nodes
ig2<- mapGraph(ig, type= "source"); gplot(ig2)

sem02 <- SEMrun(ig2, data[train,], SE="none", limit=1000)
res02<- predict(sem02, newdata=data[-train,], verbose=TRUE) 
}

}
\references{
de Rooij M, Karch JD, Fokkema M, Bakk Z, Pratiwi BC, and Kelderman H
(2023). SEM-Based Out-of-Sample Predictions, Structural Equation Modeling:
A Multidisciplinary Journal, 30:1, 132-148
<https://doi.org/10.1080/10705511.2022.2061494>

Grassi M, Palluzzi F, Tarantino B (2022). SEMgraph: An R Package for Causal Network
Analysis of High-Throughput Data with Structural Equation Models.
Bioinformatics, 38 (20), 4829–4830 <https://doi.org/10.1093/bioinformatics/btac567>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
