% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute.brt.R
\name{compute.brt}
\alias{compute.brt}
\title{Compute BRT (Boosted Regression Trees) model}
\usage{
compute.brt(x, proj.predictors, tc = 2, lr = 0.001, bf = 0.75,
           n.trees = 50, step.size = n.trees)
}
\arguments{
\item{x}{\link{SDMtab} object or dataframe that contains id, longitude, latitude and values of environmental descriptors at corresponding locations}

\item{proj.predictors}{RasterStack of environmental descriptors on which the model will be projected}

\item{tc}{Integer. Tree complexity. Sets the complexity of individual trees}

\item{lr}{Learning rate. Sets the weight applied to individual trees}

\item{bf}{Bag fraction. Sets the proportion of observations used in selecting variables}

\item{n.trees}{Number of initial trees to fit. Set at 50 by default}

\item{step.size}{Number of trees to add at each cycle}
}
\value{
\itemize{
A list of 4
\item \emph{model$algorithm} "brt" string character
\item \emph{model$data} x dataframe that was used to implement the model
\item \emph{model$response} Parameters returned by the model object
\item \emph{model$raster.prediction} Raster layer that predicts the potential species distribution}
}
\description{
Compute species distribution models with Boosted Regression Trees
}
\details{
The function realizes a BRT model according to the \link[dismo]{gbm.step} function provided by Elith et al.(2008). See the publication for further information about setting decisions. \cr The map produced provides species presence probability on the projected area.
}
\note{
See Barbet Massin et al. (2012) for information about background selection to implement BRT models
}
\examples{
\dontrun{
#Download the presence data
data('ctenocidaris.nutrix')
occ <- ctenocidaris.nutrix
# select longitude and latitude coordinates among all the information
occ <- ctenocidaris.nutrix[,c('decimal.Longitude','decimal.Latitude')]

#Download an example of environmental predictors
#restricted on geographical extent and depth (-1500m)
envi <- raster::stack(system.file('extdata', 'pred.grd',package='SDMPlay'))
envi

#Open SDMtab matrix
x <- system.file(file='extdata/SDMdata1500.csv',package='SDMPlay')
SDMdata <- read.table(x,header=TRUE, sep=';')

#Run the model
model <- SDMPlay:::compute.brt (x=SDMdata, proj.predictors=envi,lr=0.0005)

#Plot the partial dependance plots
dismo::gbm.plot(model$response)

#Get the contribution of each variable for the model
model$response$contributions

#Get the interaction between variables
dismo::gbm.interactions(model$response)
#Plot the interactions
int <- dismo::gbm.interactions(model$response)
# choose the interaction to plot
dismo::gbm.perspec(model$response,int$rank.list[1,1],int$rank.list[1,3])

#Plot the map prediction
library(grDevices) # add nice colors
palet.col <- colorRampPalette(c('deepskyblue','green','yellow', 'red'))( 80 )
raster::plot(model$raster.prediction, col=palet.col)
#add data
points(occ, col='black',pch=16)
}
}
\references{
Barbet Massin M, F Jiguet, C Albert & W Thuiller (2012) Selecting pseudo absences for species distribution models: how, where and how many? \emph{Methods in Ecology and Evolution}, 3(2): 327-338.

Elith J, J Leathwick & T Hastie (2008) A working guide to boosted regression trees. \emph{Journal of Animal Ecology}, 77(4): 802-813.
}
\seealso{
\link[dismo]{gbm.step}
}
