## Wrapper around paste for simple yet flexible namespacing of the attribute
## names. Is set to do nothing for now
myAtt <- function(att) att


## Mechanism to draw render information from a graph or from the
## defaults if not specified in the graph. The hierarchy is:
##   1. graph package defaults as set by graph.par
##   2. graph object defaults set by parRenderInfo in the renderInfo@pars slot
##   3. node or edge specific settings set by edgeRenderInfo or nodeRenderInfo
##      in slots renderInfo@edges and renderInfo@nodes
getRenderPar <-
    function(g, name, what = c("nodes", "edges", "graph"))
{
    what <- match.arg(what)
    nms <- switch(what, nodes=nodes(g),
                  edges=edgeNames(g, recipEdges=graphRenderInfo(g,
                                     "recipEdges")),
                  graph="graph") #FIXME: Deal with graph names 
    ans <- switch(what,
                  nodes = nodeRenderInfo(g, name), 
                  edges = edgeRenderInfo(g, name),
                  graph = graphRenderInfo(g, name))
    if (!is.null(ans) && !any(is.na(ans)))
        if(!is.null(names(ans)))
            ans[nms]
        else
            ans
    else
    {
        default <- parRenderInfo(g, what)[[name]][1]
        if (is.null(default)) default <- graph.par.get(what)[[name]][1]
        if (is.null(ans)) rep(default, length(nms))
        else
        {
            if(!is.null(default))
                ans[is.na(ans)] <- default
            ans[nms]
        }
    }
}



## This function will plot individual nodes on the plotting device.
## Update: This is now in a vectorized form (user can still
## supply a function, but that has to deal with vectorized data for now)
renderNodes <- function(g) 
{
    ## get necessary render parameters from the graph or use defaults
    ## these are generated by the layout algorithm
    nodeX <- getRenderPar(g, "nodeX", "nodes")
    nodeY <- getRenderPar(g, "nodeY", "nodes")
    lw <- getRenderPar(g, "lWidth", "nodes")
    rw <- getRenderPar(g, "rWidth", "nodes")
    height <- getRenderPar(g, "height", "nodes")
    rad    <- (lw+rw)/2
    labelX <- getRenderPar(g, "labelX", "nodes")
    labelY <- getRenderPar(g, "labelY", "nodes")
    #labelJust <- getRenderPar(g, "labelJust", "nodes") ## FIXME: do we need this
    #labelJust <- as.numeric(gsub("l", 0, gsub("n", -0.5, gsub("r", -1,
    #                        labelJust))))
    ## these only live within R
    fill <- getRenderPar(g, "fill", "nodes")
    col <- getRenderPar(g, "col", "nodes")
    lwd <- getRenderPar(g, "lwd", "nodes")
    lty <- getRenderPar(g, "lty", "nodes")
    textCol <- getRenderPar(g, "textCol", "nodes")
    style <- getRenderPar(g, "style", "nodes")
    shape <- getRenderPar(g, "shape", "nodes") 
    label <- getRenderPar(g, "label", "nodes")
    fontsize <- getRenderPar(g, "fontsize", "nodes")
    if (is.null(label)) label <- nodes(g)
   

    ## deal with different shapes
    possible.shapes <-
        c("circle", "ellipse", "box", "rectangle", "plaintext", "triangle")
    shape <-
        possible.shapes[pmatch(shape,
                               possible.shapes,
                               duplicates.ok = TRUE)]
    ## shape == circle
    i <- shape == "circle"
    if (any(i, na.rm=TRUE))
    {
        symbols(nodeX[i], nodeY[i], circles = rad[i],
                fg = col[i], bg = fill[i], lwd = lwd[i], lty = lty[i],
                inches = FALSE, add = TRUE)
    }
    ## shape == box, rect, etc
    i <- shape %in% c("box", "rectangle", "rect")
    if (any(i, na.rm=TRUE))
    {
        rect(nodeX[i] - lw[i], nodeY[i] - (height[i] / 2),
             nodeX[i] + rw[i], nodeY[i] + (height[i] / 2),
             col = fill[i], border = col[i], lty = lty[i], lwd = lwd[i])
    }
    ## shape == triangle
    ## FIXME: The edges are not computed for triangle shapes in Graphviz
    ##        allthough the correct shape is stored in the agraph object.
    ##        There must be something weird going on internally in the
    ##        C code....
    i <- shape == "triangle"
    if (any(i, na.rm=TRUE))
    {
        polygon(x=c(nodeX[i] - lw[i], nodeX[i], nodeX[i] + lw[i]),
                y=c(nodeY[i] - (height[i] / 2), nodeY[i] + (height[i] / 2),
                nodeY[i] - (height[i] / 2)),
                col = fill[i], border = col[i], lty = lty[i], lwd = lwd[i])
    }
    
    ## shape == ellipse
    i <- shape == "ellipse"
    if (any(i, na.rm=TRUE))
    {
        npoints <- 101
        tt <- c(seq(-pi, pi, length = npoints), NA)
        xx <-
            rep(nodeX[i], each = npoints + 1) +
                sin(tt) * rep(rad[i], each = npoints + 1)
        yy <-
            rep(nodeY[i], each = npoints + 1) +
                cos(tt) * rep(height[i] / 2, each = npoints + 1)
        polygon(xx, yy, border = col[i], col = fill[i], lwd = lwd[i],
                lty = lty[i])
    }
    
    ## shape == plaintext
    ## nothing to do (for style = "filled", use fill = "grey")
    
    ## compute label cex from node dimensions if not set
    cex <- getRenderPar(g, "cex", "nodes")
    if(is.null(cex)){
        nodeDims <- cbind(lw+rw, height)
        stw <- strwidth(label)
        sth <- strheight(label)
        strDims  <- cbind(stw*1.1, sth*1.4)
        strDims[!nzchar(label),] <- c(strwidth(" "), strheight(" "))
        cex <- min(nodeDims / strDims)
    }
    
    ## draw labels
    text(labelX, labelY, label, col=textCol,
         cex=cex*as.numeric(fontsize)/14)
}



## A vectorized function that draws the splines for the edges
renderSpline <-
    function(spline, head = FALSE, tail = FALSE, len = 1,
             col = "black", lwd=1, lty="solid", ...)
{
    ## may get numerics as characters (e.g. "1") which doesn't work
    ## for 'lines'
    mylty <- as.numeric(lty)
    if(!is.na(mylty)) lty <- mylty
    lapply(spline, lines, col = col, lwd=lwd, lty=lty, ...)
    
    ## the arrow heads
    if (head)
    {
        xy <- tail(bezierPoints(spline[[length(spline)]]), 2)
        arrows(xy[1], xy[3], xy[2], xy[4], length = len, col = col,
               lwd=lwd, lty=lty)
    }
    if (tail)
    {
        xy <- head(bezierPoints(spline[[1]]), 2)
        arrows(xy[2], xy[4], xy[1], xy[3], length = len, col = col,
               lwd=lwd, lty=lty)
    }
}



## find R's resolution for the current device
devRes <- function(){
    require(grid)
    if(current.viewport()$name != "ROOT"){
        vpt <- current.vpTree()
        popViewport(0)
        xres <- abs(as.numeric(convertWidth(unit(1, "inches"), "native")))
        yres <- abs(as.numeric(convertHeight(unit(1, "inches"), "native")))
    pushViewport(vpt)
    }else{
        xres <- abs(as.numeric(convertWidth(unit(1, "inches"), "native")))
        yres <- abs(as.numeric(convertHeight(unit(1, "inches"), "native")))
    }
    retval <- c(xres, yres)
    names(retval) <- c("xres", "yres")
    return(retval)
}



## This function will plot individual edges on the plotting device.
renderEdges <- function(g)
{
    ## get necessary render parameters
    ## these are generated by the layout algorithm
    lw <- getRenderPar(g, "lWidth", "nodes")
    rw <- getRenderPar(g, "rWidth", "nodes")
    height <- getRenderPar(g, "height", "nodes")
    splines <- getRenderPar(g, "splines", "edges")
    ## direction <- getRenderPar(g, "direction", "edges") ## UNUSED (isn't this redundant?)
    arrowhead <- getRenderPar(g, "arrowhead", "edges") != "none"
    arrowtail <- getRenderPar(g, "arrowtail", "edges") != "none"
    label <- getRenderPar(g, "label", "edges")
    labelX <- getRenderPar(g, "labelX", "edges")
    labelY  <- getRenderPar(g, "labelY", "edges")
    #labelJust <- getRenderPar(g, "labelJust", "edges") ## FIXME:do we need this
    #labelJust <- as.numeric(gsub("l", 0, gsub("n", -0.5, gsub("r", -1,
    #                        labelJust))))              
    #labelWidth <- getRenderPar(g, "labelWidth", "edges")
    ## these only live within R
    fontsize <- getRenderPar(g, "fontsize", "edges")
    textCol <- getRenderPar(g, "textCol", "edges")
    col <- getRenderPar(g, "col", "edges")
    lty <- getRenderPar(g, "lty", "edges")
    lwd <- getRenderPar(g, "lwd", "edges")
    cex <- getRenderPar(g, "cex", "edges")
    
    ## set the arrow size
    minDim <- min(rw + lw, height)
    arrowLen <- par("pin")[1] / diff(par("usr")[1:2]) * minDim / (1.5*pi)

    ## plot the edge splines
    for (i in seq_along(splines))
    {
        suppressWarnings(renderSpline(splines[[i]],
                                      head = arrowhead[i],
                                      tail = arrowtail[i],
                                      len = arrowLen,
                                      col = col[i], lty = lty[i],
                                      lwd = lwd[i]))
    }

    ## draw text labels
    text(labelX, labelY, label, col=textCol,
         cex=cex*as.numeric(fontsize)/14)
}



## render graph to plotting device
setGeneric("renderGraph",
           function(x, ...) standardGeneric("renderGraph"))

setMethod("renderGraph", "graph",
          function(x, ...,
                   drawNodes = "renderNodes",
                   drawEdges = renderEdges,
                   graph.pars=list())
      {

          ## evaluate defaults passed in via the graph.pars argument
          old.graph.pars <- graph.par(graph.pars)
          on.exit(graph.par(old.graph.pars))

          ## check that the graph has been laid out
          laidout <- getRenderPar(x, "laidout", "graph")
          bbox <- getRenderPar(x, "bbox", "graph")
          if(!laidout)
              stop("Graph has not been laid out yet. Please use function ",
                   "'layoutGraph'")
          plot.new()

          ## eliminate all plot borders but leave space for title and
          ## subtitle if needed
          sub <-  getRenderPar(x, "sub", "graph")
          main <- getRenderPar(x, "main", "graph")
          cex.main <- getRenderPar(x, "cex.main", "graph")
          cex.sub <- getRenderPar(x, "cex.sub", "graph")
          mheight <- if(!is.null(main) && nchar(main)>0)
              strheight(main, "inches", cex.main)+0.3 else 0.1
          sheight <- if(!is.null(sub) && nchar(sub)>0)
              strheight(sub, "inches", cex.sub)+0.2 else 0.1
          old.pars <- par(mai=c(sheight, 0, mheight,0))
          on.exit(par(old.pars), add=TRUE)

          ## set coordinate system to the values of the bounding box
          ## and keep aspect ratio fixed when margins increase due to
          ## title and subtitle
          aspFact <- (sheight+mheight)/par("din")[2]
          usr <- c(bbox[1,1] - (bbox[2,1] * (aspFact/2)),
                    bbox[2,1] + (bbox[2,1] * (aspFact/2)),
                    bbox[,2])
          plot.window(xlim=usr[1:2], ylim=usr[3:4],
                      log="", asp=NA)
          old.pars <- append(old.pars, par(usr=usr))

          ## Add title and subtitle if available
          old.pars <- append(old.pars, par(xpd=NA))
          if(mheight>0.1){
              col.main <- getRenderPar(x, "col.main", "graph")
              moffset <- (bbox[2,2]/par("pin")[2] * mheight)/2
              text(bbox[2,1]/2, bbox[2,2] + moffset, main,
                   cex=cex.main, col=col.main, adj=c(0.5))
          }
          if(sheight>0.1){
              col.sub<- getRenderPar(x, "col.sub", "graph")
              soffset <- (bbox[2,2]/par("pin")[2] * sheight)/2
              text(bbox[2,1]/2, bbox[1,2] - soffset,
                   sub, cex=cex.sub, col=col.sub, adj=c(0.5))
          }
          
          ## Draw Nodes, using default vectorized function or a
          ## node-by-node user-defined function   
          if(is.character(drawNodes)){
                  if(match.arg(drawNodes)=="renderNodes")
                    Rgraphviz:::renderNodes(x)
              }else  drawNodes(x)


          ## Draw edges using default edge rendering function
          drawEdges(x)

          ## compute native node coordinates for imageMaps
          x1 <- {getRenderPar(x, "nodeX", "nodes") -
                     getRenderPar(x, "lWidth", "nodes")}
          y1 <- {getRenderPar(x, "nodeY", "nodes") -
                     getRenderPar(x, "height", "nodes")/2}
          x2 <- {getRenderPar(x, "nodeX", "nodes") +
                     getRenderPar(x, "rWidth", "nodes")}
          y2 <- {getRenderPar(x, "nodeY", "nodes") +
                     getRenderPar(x, "height", "nodes")/2}
          figDims <- par("din")
          ## these factors should accomodate for any figure margins
          xfac <- diff(par("plt")[1:2])
          xoffset <- par("plt")[1]
          yfac <- diff(par("plt")[3:4])
          yoffset <- par("plt")[3]
          ## need to take into account the aspect factor for x values
          x1n <- {((x1/diff(usr[1:2])) * xfac) + xoffset +
                      (bbox[1,1]-usr[1])/diff(usr[1:2])}
          x2n <- {((x2/diff(usr[1:2])) * xfac) + xoffset +
                      (bbox[1,1]-usr[1])/diff(usr[1:2])}
          ## invert y values because [0,0] is on top left for imageMap
          y1n <- 1-(((y1/bbox[2,2])*yfac)+yoffset)
          y2n <- 1-(((y2/bbox[2,2])*yfac)+yoffset)
          nativeCoords <- cbind(x1n, y1n, x2n,y2n)

          ## store information about the rendering process in the graph
          graphRenderInfo(x) <- list(nativeCoords=nativeCoords,
                                     figDim=figDims*devRes(),
                                     usr=usr, mai=par("mai"))
          
          return(invisible(x))
      })









#############################################################################
##---------------------------------------------------------------------------
#############################################################################
## THE ATTIC
#############################################################################
##---------------------------------------------------------------------------
#############################################################################


## This function draws the node and edge labels on the plotting device.
## It is called by myDrawAgNode. The input is again a list of nodeData## as well as the x and y location of the nodes.
## FIXME: This doesn't have to be called every time by myDrawAgNode. It could be
## a vectorized version of text that gets called after the node plotting.
myDrawTxtLabel <- function(attr, xLoc, yLoc) {
  txt <- attr[[myAtt("label")]]

  if(length(txt)>1) stop("label must have length 1.")

  if(length(txt)==0 || txt=="") return(invisible(NULL))

  if(xor(missing(xLoc), missing(yLoc)))
    stop("'xLoc' and 'yLoc' must be either be both specified or both missing.")
  ## When there's no x location it is computed according to the adjustment
  if(missing(xLoc)) {
    lj <- attr[[myAtt("labelJust")]]
    if(length(lj)){
      justMod <- switch(lj,
                        "l" = 0,
                        "n" = -0.5,
                        "r" = -1)
      xLoc <-   as.numeric(attr[[myAtt("labelX")]]) +(justMod * as.numeric(attr[[myAtt("labelWidth")]]))
      yLoc <-  as.numeric(attr[[myAtt("labelY")]])
    }else{
      xLoc <- yLoc <- 0
    }
  }
  ## NOTE: labelFontsize is translated into cex parameter: fontsize 14 = cex 1
  text(xLoc, yLoc, txt, col=attr[[myAtt("fontcolor")]],
       cex=as.numeric(attr[[myAtt("fontsize")]])/14)
}
