% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fparse.R, R/fload.R
\name{fparse}
\alias{fparse}
\alias{fload}
\title{Fast, Friendly, and Flexible JSON Parsing}
\usage{
fparse(
  json,
  query = "",
  empty_array = NULL,
  empty_object = NULL,
  single_null = NULL,
  error_ok = FALSE,
  on_error = NULL,
  max_simplify_lvl = c("data_frame", "matrix", "vector", "list"),
  type_policy = c("anything_goes", "numbers", "strict"),
  int64_policy = c("double", "string", "integer64")
)

fload(
  json,
  query = "",
  empty_array = NULL,
  empty_object = NULL,
  single_null = NULL,
  error_ok = FALSE,
  on_error = NULL,
  max_simplify_lvl = c("data_frame", "matrix", "vector", "list"),
  type_policy = c("anything_goes", "numbers", "strict"),
  int64_policy = c("double", "string", "integer64"),
  verbose = FALSE,
  temp_dir = tempdir(),
  keep_temp_files = FALSE
)
}
\arguments{
\item{json}{One or more \code{character}s of JSON or paths to files containing JSON.}

\item{query}{String used as a JSON Pointer to identify a specific element within \code{json}.
\code{character(1L)}, default: \code{""}}

\item{empty_array}{Any R object to return for empty JSON arrays.
default: \code{NULL}}

\item{empty_object}{Any R object to return for empty JSON objects.
default: \code{NULL}.}

\item{single_null}{Any R object to return for single JSON nulls.
default: \code{NULL}.}

\item{error_ok}{Whether to allow parsing errors.
default: \code{FALSE}.}

\item{on_error}{If \code{error_ok} is \code{TRUE}, \code{on_error} is any
R object to return when parsing errors occur.
default: \code{NULL}.}

\item{max_simplify_lvl}{Maximum simplification level.
 \code{character(1L)} or \code{integer(1L)}, default: \code{"data_frame"}
 \itemize{
   \item \code{"data_frame"} or \code{0L}
   \item \code{"matrix"} or \code{1L}
   \item \code{"vector"} or \code{2L}
   \item \code{"list"} or \code{3L} (no simplification)
}}

\item{type_policy}{Level of type strictness.
\code{character(1L)} or \code{integer(1L)}, default: \code{"anything_goes"}.
\itemize{
  \item \code{"anything_goes"} or \code{0L}: non-recursive arrays always become atomic vectors
  \item \code{"numbers"} or \code{1L}: non-recursive arrays containing only numbers always become atomic vectors
  \item \code{"strict"} or \code{2L}: non-recursive arrays containing mixed types never become atomic vectors
 }}

\item{int64_policy}{How to return big integers to R.
\code{character(1L)} or \code{integer(1L)}, default: \code{"double"}.
\itemize{
  \item \code{"double"} or \code{0L}: big integers become \code{double}s
  \item \code{"string"} or \code{1L}: big integers become \code{character}s
  \item \code{"integer64"} or \code{2L}: big integers \code{bit64::integer64}s
}}

\item{verbose}{Whether to display status messages.
\code{TRUE} or \code{FALSE}, default: \code{FALSE}}

\item{temp_dir}{Directory path to use for any temporary files.
\code{character(1L)}, default: \code{tempdir()}}

\item{keep_temp_files}{Whether to remove any temporary files created by
\code{fload()} from \code{temp_dir}.
\code{TRUE} or \code{FALSE}, default: \code{TRUE}}
}
\description{
Parse JSON strings and files to R objects.
}
\details{
\itemize{

  \item Instead of using \code{lapply()} to parse multiple values, just use
  \code{fparse()} and \code{fload()} directly.
  \itemize{
    \item They are vectorized in order to leverage the underlying
          \code{simdjson::dom::parser}'s ability to reuse its internal buffers
           between parses.
    \item Since the overwhelming majority of JSON parsed will not result in
          scalars, a \code{list()} is always returned if \code{json} contains
          more than one value.
    \item If \code{json} contains multiple values and has \code{names()}, the
          returned object will have the same names.
    \item If \code{json} contains multiple values and is unnamed, \code{fload()}
          names each returned element using the file's \code{basename()}.
   }

}
}
\examples{
# simple parsing ============================================================
json_string <- '{"a":[[1,null,3.0],["a","b",true],[10000000000,2,3]]}'
fparse(json_string)

# controlling type-strictness ===============================================
fparse(json_string, type_policy = "numbers")
fparse(json_string, type_policy = "strict")
fparse(json_string, type_policy = "numbers", int64_policy = "string")

if (requireNamespace("bit64", quietly = TRUE)) {
    fparse(json_string, type_policy = "numbers", int64_policy = "integer64")
}

# vectorized parsing ========================================================
json_strings <- c(
    json1 = '[{"b":true,
               "c":null},
              {"b":[[1,2,3],
                    [4,5,6]],
               "c":"Q"}]',
    json2 = '[{"b":[[7, 8, 9],
                   [10,11,12]],
              "c":"Q"},
             {"b":[[13,14,15],
                   [16,17,18]],
              "c":null}]'
)
fparse(json_strings)

# controlling simplification ================================================
fparse(json_strings, max_simplify_lvl = "matrix")
fparse(json_strings, max_simplify_lvl = "vector")
fparse(json_strings, max_simplify_lvl = "list")

# customizing what `[]`, `{}`, and single `null`s return ====================
empties <- "[[],{},null]"
fparse(empties)
fparse(empties,
       empty_array = logical(),
       empty_object = `names<-`(list(), character()),
       single_null = NA_real_)

# handling invalid JSON and parsing errors ==================================
fparse("junk JSON", error_ok = TRUE)
fparse("junk JSON", error_ok = TRUE, on_error = "can't parse invalid JSON")
fparse(
    c(junk_JSON_1 = "junk JSON 1",
      valid_JSON_1 = '"this is valid JSON"',
      junk_JSON_2 = "junk JSON 2",
      valid_JSON_2 = '"this is also valid JSON"'),
    error_ok = TRUE,
    on_error = NA
)

# querying JSON w/ a JSON Pointer ===========================================
json_to_query <- c(
    json1 = '[
    "a",
    {
        "b": {
            "c": [
                [
                    1,
                    2,
                    3
                ],
                [
                    4,
                    5,
                    6
                ]
            ]
        }
    }
]',
    json2 = '[
    "a",
    {
        "b": {
            "c": [
                [
                    7,
                    8,
                    9
                ],
                [
                    10,
                    11,
                    12
                ]
            ]
        }
    }
]')
fparse(json_to_query, query = "1")
fparse(json_to_query, query = "1/b")
fparse(json_to_query, query = "1/b/c")
fparse(json_to_query, query = "1/b/c/1")
fparse(json_to_query, query = "1/b/c/1/0")

# load JSON files ===========================================================
single_file <- system.file("jsonexamples/small/demo.json", package = "RcppSimdJson")
fload(single_file)

multiple_files <- c(
  single_file,
  system.file("jsonexamples/small/smalldemo.json", package = "RcppSimdJson")
)
fload(multiple_files)

# load remote JSON ==========================================================
\dontrun{

a_url <- "https://api.github.com/users/lemire"
fload(a_url)

multiple_urls <- c(
  a_url,
  "https://api.github.com/users/eddelbuettel",
  "https://api.github.com/users/knapply",
  "https://api.github.com/users/dcooley"
)
fload(multiple_urls, query = "name", verbose = TRUE)
}

}
\author{
Brendan Knapp
}
