% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cloudMask.R
\name{cloudMask}
\alias{cloudMask}
\title{Simple Cloud Detection}
\usage{
cloudMask(x, threshold = 0.8, blue = "B1_sre", tir = "B6_sre",
  buffer = NULL, plot = FALSE, verbose)
}
\arguments{
\item{x}{RasterBrick or RasterStack with reflectance and brightness temperature OR the mask of a previous run of \code{cloudMask} with \code{returnDiffLayer=TRUE}.}

\item{threshold}{Numeric. cloud detection threshold. If not provided it will be guessed. Everything *below* this threshold will be considered a cloud pixel (unless it is removed by filtering afterwards).}

\item{blue}{Character or integer. Bandname or number for the blue band}

\item{tir}{Character or integer. Bandname or number for the thermal band}

\item{buffer}{Integer. Number of pixels to use as a buffer that will be added to the identified cloud centers.}

\item{plot}{Logical. Plots of the cloud mask for all sub-steps (sanitizing etc.) Helpful to find proper parametrization.}

\item{verbose}{Logical. Print messages or suppress.}
}
\value{
Returns a RasterStack with two layers: CMASK contains the binary cloud mask (1 = cloud, NA = not-cloud) and NDTCI contains the cloud index.
}
\description{
Developed for use with Landsat data \code{cloudMask} relies on the distinctive difference between the blue (or any other short-wave band) 
and thermal band for semi-automated creation of a cloud mask. Since it relies on thermal information it doesn't work well for sensors without
thermal bands.
}
\note{
Typically clouds are cold in the thermal region and have high reflectance in short wavelengths (blue). By calculating a normalized difference index between the two bands and thresholding a rough cloud mask can be obtained.
Before calculating the spectral cloud index (let's call it Normalized Difference Thermal Cloud Index (NDTCI)) the thermal band will be matched to the same value range as the blue band. Therefore, it doesn't matter whether you
provide DN, radiance or brightness temperature.

This approach to cloud masking is very simplistic. And aims only at rough removal of potentially clouded areas. Nevertheless, it is able to provide satisfactory results. 
More sophisticated approaches, including cloud cast shadow detection can be found elsewhere, e.g. \href{http://code.google.com/p/fmask}{fmask}.

It can make sense to find a suitable threshold on a cropped version of the scene. Also make sure you make use of the \code{returnDiffLayer} argument to save yourself one processing step.
Buffering should be seen as final polishing, i.e. as long as the pure cloud centers are not detected properly, you might want to turn it off. since it takes some time to calculate.
Once your mask detects obvious cloud pixels properly re-enable buffering for fine tuning if desired. Finally, once a suitable threshold is established re-run cloudMask on the whole scene with this threshold and go get a coffee.
}
\examples{
  
library(raster)
library(ggplot2)
## Import Landsat example subset
data(lsat) 
## We have two tiny clouds in the east
\donttest{ggRGB(lsat, stretch = "lin")}

## Calculate cloud index
cldmsk    <- cloudMask(lsat, blue = 1, tir = 6)
\donttest{ggR(cldmsk, 2, geom_raster = TRUE) }

## Define threshold (re-use the previously calculated index)
## Everything above the threshold is masked
## In addition we apply a region-growing around the core cloud pixels
cldmsk_final <- cloudMask(cldmsk, threshold = 0.1, buffer = 5) 

## Plot cloudmask 
\donttest{ggRGB(lsat, stretch = "lin") +
   ggR(cldmsk_final[[1]], ggLayer = TRUE, forceCat = TRUE, geom_raster = TRUE) +
   scale_fill_manual(values = "red", na.value = NA)
}
#' ## Estimate cloud shadow displacement
## Interactively (click on cloud pixels and the corresponding shadow pixels)
\dontrun{ shadow <- cloudShadowMask(lsat, cldmsk_final, nc = 2) }

## Non-interactively. Pre-defined shadow displacement estimate (shiftEstimate)
\donttest{shadow <- cloudShadowMask(lsat, cldmsk_final, shiftEstimate = c(-16,-6))}

## Plot
\donttest{csmask <- raster::merge(cldmsk_final[[1]], shadow)
ggRGB(lsat, stretch = "lin") +
        ggR(csmask, ggLayer = TRUE, forceCat = TRUE, geom_raster = TRUE) +
        scale_fill_manual(values = c("blue", "yellow"), 
        labels = c("shadow", "cloud"), na.value = NA)
}
}
\seealso{
\code{\link{cloudShadowMask}}
}
