% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RNAmf_three_level.R
\name{RNAmf_three_level}
\alias{RNAmf_three_level}
\title{Fitting the model with three fidelity levels}
\usage{
RNAmf_three_level(X1, y1, X2, y2, X3, y3, kernel = "sqex", constant = TRUE, ...)
}
\arguments{
\item{X1}{vector or matrix of input locations for the low fidelity level.}

\item{y1}{vector of response values for the low fidelity level.}

\item{X2}{vector or matrix of input locations for the medium fidelity level.}

\item{y2}{vector of response values for the medium fidelity level.}

\item{X3}{vector or matrix of input locations for the high fidelity level.}

\item{y3}{vector of response values for the high fidelity level.}

\item{kernel}{character specifying kernel type to be used, to be chosen between \code{"sqex"}(squared exponential), \code{"matern1.5"}, or \code{"matern2.5"}. Default is \code{"sqex"}.}

\item{constant}{logical indicating for constant mean of GP (\code{constant=TRUE}) or zero mean (\code{constant=FALSE}). Default is \code{TRUE}.}

\item{...}{for compatibility with \code{optim}.}
}
\value{
\itemize{
  \item \code{fit.RNAmf_two_level}: a class \code{RNAmf} object fitted by \code{RNAmf_two_level}. It contains a list of \eqn{\begin{cases} & \text{\code{fit1} for } (X_1, y_1),\\ & \text{\code{fit2} for } ((X_2, f_1(X_2)), y_2), \end{cases}}. See \code{\link{RNAmf_two_level}}.
  \item \code{fit3}: list of fitted model for \eqn{((X_2, f_2(X_3, f_1(X_3))), y_3)}.
  \item \code{constant}: copy of \code{constant}.
  \item \code{kernel}: copy of \code{kernel}.
  \item \code{level}: a level of the fidelity. It returns 3.
  \item \code{time}: a scalar of the time for the computation.
}
}
\description{
The function fits RNA models with designs of three fidelity levels.
The estimation method is based on MLE.
Possible kernel choices are squared exponential, Matern kernel with smoothness parameter 1.5 and 2.5.
The function returns fitted model by \code{\link{RNAmf_two_level}}, fitted model at level 3, whether constant mean or not, and kernel choice.
}
\details{
Consider the model
\eqn{\begin{cases}
& f_1(\bm{x}) = W_1(\bm{x}),\\
& f_l(\bm{x}) = W_l(\bm{x}, f_{l-1}(\bm{x})) \quad\text{for}\quad l=2,3,
\end{cases}}
where \eqn{f_l} is the simulation code at fidelity level \eqn{l}, and
\eqn{W_l(\bm{x}) \sim GP(\alpha_l, \tau_l^2 K_l(\bm{x}, \bm{x}'))} is GP model.
Hyperparameters \eqn{(\alpha_l, \tau_l^2, \bm{\theta_l})} are estimated by
maximizing the log-likelihood via an optimization algorithm "L-BFGS-B".
For \code{constant=FALSE}, \eqn{\alpha_l=0}.

Covariance kernel is defined as:
\eqn{K_l(\bm{x}, \bm{x}')=\prod^d_{j=1}\phi(x_j,x'_j;\theta_{lj})} with
\eqn{\phi(x, x';\theta) = \exp \left( -\frac{ \left( x - x' \right)^2}{\theta}  \right)}
for squared exponential kernel; \code{kernel="sqex"},
\eqn{\phi(x,x';\theta) =\left( 1+\frac{\sqrt{3}|x- x'|}{\theta} \right) \exp \left( -\frac{\sqrt{3}|x- x'|}{\theta} \right)}
for Matern kernel with the smoothness parameter of 1.5; \code{kernel="matern1.5"} and
\eqn{\phi(x, x';\theta) = \left( 1+\frac{\sqrt{5}|x-x'|}{\theta} +\frac{5(x-x')^2}{3\theta^2} \right) \exp \left( -\frac{\sqrt{5}|x-x'|}{\theta} \right)}
for Matern kernel with the smoothness parameter of 2.5; \code{kernel="matern2.5"}.

For details, see Heo and Sung (2024, <\doi{https://doi.org/10.1080/00401706.2024.2376173}>).
}
\examples{
\donttest{
### three levels example ###
library(lhs)

### Branin function ###
branin <- function(xx, l){
  x1 <- xx[1]
  x2 <- xx[2]
  if(l == 1){
    10*sqrt((-1.275*(1.2*x1+0.4)^2/pi^2+5*(1.2*x1+0.4)/pi+(1.2*x2+0.4)-6)^2 +
    (10-5/(4*pi))*cos((1.2*x1+0.4))+ 10) + 2*(1.2*x1+1.9) - 3*(3*(1.2*x2+2.4)-1) - 1 - 3*x2 + 1
  }else if(l == 2){
    10*sqrt((-1.275*(x1+2)^2/pi^2+5*(x1+2)/pi+(x2+2)-6)^2 +
    (10-5/(4*pi))*cos((x1+2))+ 10) + 2*(x1-0.5) - 3*(3*x2-1) - 1
  }else if(l == 3){
    (-1.275*x1^2/pi^2+5*x1/pi+x2-6)^2 + (10-5/(4*pi))*cos(x1)+ 10
  }
}

output.branin <- function(x, l){
  factor_range <- list("x1" = c(-5, 10), "x2" = c(0, 15))

  for(i in 1:length(factor_range)) x[i] <- factor_range[[i]][1] + x[i] * diff(factor_range[[i]])
  branin(x[1:2], l)
}

### training data ###
n1 <- 20; n2 <- 15; n3 <- 10

### fix seed to reproduce the result ###
set.seed(1)

### generate initial nested design ###
X <- NestedX(c(n1, n2, n3), 2)
X1 <- X[[1]]
X2 <- X[[2]]
X3 <- X[[3]]

### n1, n2 and n3 might be changed from NestedX ###
### assign n1, n2 and n3 again ###
n1 <- nrow(X1)
n2 <- nrow(X2)
n3 <- nrow(X3)

y1 <- apply(X1,1,output.branin, l=1)
y2 <- apply(X2,1,output.branin, l=2)
y3 <- apply(X3,1,output.branin, l=3)

### fit an RNAmf ###
fit.RNAmf <- RNAmf_three_level(X1, y1, X2, y2, X3, y3, kernel = "sqex")
}


}
\seealso{
\code{\link{predict.RNAmf}} for prediction.
}
