#' Search and download MODIS images
#'
#' \code{modDownSearch} searches and downloads MODIS images concerning a particular
#' location and time interval from the `EarthData' repository. Images are
#' saved as GTiff files in the \code{AppRoot} directory.
#'
#' \code{modDownSearch} searches via 
#' \href{https://lpdaacsvc.cr.usgs.gov/services/inventory}{NASA’s Common Metadata Repository}
#' and downloads the imagert from the
#' \href{https://earthdata.nasa.gov/}{`EarthData' web service}
#' to download the imagery. The catalogue of MODIS products can be found
#' \href{https://modis.gsfc.nasa.gov/data/dataprod/}{here}.
#' The catalogue shows detailed information about the products and their short
#' names. By the time `RGISTools' is released, NASA carries out the maintenance
#' of its website on Wednesdays, which may cause an error when connecting to
#' their server. You can get your `EarthData' credentials
#' \href{https://urs.earthdata.nasa.gov/users/new}{here}.
#'
#' @param product a \code{character} argument with the short name of the MODIS
#' product.

#' @param username NASA's `EarthData' username.
#' @param password NASA's `EarthData' password.
#' @param AppRoot the directory to save the outcoming time series.
#' @param nattempts the number of attempts to download an image in case it
#' becomes corrupted.
#' @param collection MODIS collection, by default 6.
#' @param extract.tif logical argument. If \code{TRUE}, extracts all the layers
#' from hdf files and saves them as GTiff.
#' @param verbose logical argument. If \code{TRUE}, the function prints running stages and warnings.
#' @param ... arguments for nested functions:
#' \itemize{
#'   \item \code{dates} a vector with the capturing dates being searched. This
#'   argument is mandatory if \code{startDate} and \code{endDate} are not defined.
#'   \item  startDate a \code{Date} class object with the starting date of the 
#' study period. This argument is mandatory if 
#'   \code{dates} is not defined.
#'   \item  endDate a \code{Date} class object with the ending date of the 
#' study period. This argument is mandatory if 
#'   \code{dates} is not defined.
#'   \item \code{lonlat} a vector with the longitude/latitude
#'   coordinates of the point of interest. This argument is mandatory if
#'   \code{region} or \code{extent} are not defined.
#'   \item \code{extent} an \code{sf}, \code{Raster*}, or \code{Spatial*}
#'   object representing the region of interest with longitude/latitude
#'   coordinates. This argument is mandatory if \code{polygon} or \code{lonlat}
#'   are not defined.
#'   \item \code{region} A list of vectors defining the points of a polygon in
#'   longitude/latitude format. This argument is mandatory if \code{lonlat} or
#'   \code{extent} are not defined.
#'   \item Any argument in \code{\link{modExtractHDF}} function. Ex.
#'   \code{bFilter="b01_1"}.
#' }
#' @return this function does not return anything. It saves the imagery as
#' `tar.gz’ (and GTiff files) in a folder called `raw’ (`tif’) in the
#'  \code{AppRoot} directory.
#' @examples
#' \dontrun{
#' # load a spatial polygon object of Navarre
#' data(ex.navarre)
#' wdir <- file.path(tempdir(),"Path_for_downloading_folder")
#' print(src)
#' modDownSearch(product = "MOD09GA",
#'               startDate = as.Date("01-01-2018", "%d-%m-%Y"),
#'               endDate = as.Date("03-01-2018", "%d-%m-%Y"),
#'               username = "username",
#'               password = "password",
#'               AppRoot = wdir,
#'               extract.tif = TRUE,
#'               collection = 6,
#'               extent = ex.navarre)
#' wdir.mod.tif <- file.path(wdir,"Modis","MOD09GA","tif")
#' files.mod <- list.files(wdir.mod.tif,
#'                         pattern = "\\.tif$",
#'                         full.names = TRUE,
#'                         recursive = TRUE)[c(16,19,18)]
#' img.mod <- stack(files.mod)
#' qrange <- c(0.001, 0.999)
#' img.mod.rgb <- varRGB(img.mod[[1]], 
#'                       img.mod[[2]],
#'                       img.mod[[3]],
#'                       qrange)
#' plotRGB(img.mod.rgb)
#'}
modDownSearch<-function(product,
                     username,
                     password,
                     AppRoot,
                     collection=6,
                     nattempts=5,
                     verbose=FALSE,
                     extract.tif=FALSE,
                     ...){
  arg<-list(...)
  if(missing(username)|missing(password))stop("username and/or password not defined!")
  search.res<-modSearch(product=product,
                        collection=collection,
                        ...)
  if(verbose){
    message(paste0(search.res,sep="\n"))
  }
  AppRoot<-file.path(AppRoot,"Modis")
  downdir<-file.path(AppRoot,product,"hdf")
  tiffdir<-file.path(AppRoot,product,"tif")
  if(extract.tif)
    dir.create(tiffdir,recursive=TRUE,showWarnings = FALSE)
  natps<-0
  dir.create(downdir,recursive = TRUE,showWarnings = FALSE)
  for(s in search.res){
    recursiveModDownload(s=s,
                         username=username,
                         password=password,
                         downdir=downdir,
                         tiffdir=tiffdir,
                         verbose=verbose,
                         extract.tif=extract.tif,
                         nattempts=nattempts,
                         natps=0,
                         ...)
  }
  if(extract.tif){
    message(paste0("The images have been downloaded and saved on HDD. \nFile path: ",tiffdir))
  }else{
    message(paste0("The images have been downloaded and saved on HDD. \nFile path: ",downdir))
  }
}
