% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MellinQF_ratio.R
\name{compute_MellinQF_ratio}
\alias{compute_MellinQF_ratio}
\title{Mellin Transform of the Independent Positive QFs Ratio}
\source{
Ruben, Harold. "Probability content of regions under spherical normal distributions, IV:
The distribution of homogeneous and non-homogeneous quadratic functions of normal variables."
The Annals of Mathematical Statistics 33.2 (1962): 542-570.
}
\usage{
compute_MellinQF_ratio(
  lambdas_num,
  lambdas_den,
  etas_num = rep(0, length(lambdas_num)),
  etas_den = rep(0, length(lambdas_den)),
  eps = 1e-06,
  rho = 1 - 1e-04,
  maxit_comp = 1e+05,
  eps_quant = 1e-06,
  maxit_quant = 10000,
  lambdas_tol = NULL
)
}
\arguments{
\item{lambdas_num}{vector of positive weights for the numerator.}

\item{lambdas_den}{vector of positive weights for the denominator.}

\item{etas_num}{vector of non-centrality parameters for the numerator. Default all zeros (central chi square).}

\item{etas_den}{vector of non-centrality parameters for the denominator. Default all zeros (central chi square).}

\item{eps}{required absolute error for density and cumulative functions.}

\item{rho}{distribution total probability mass for which it is desired to keep the error \code{eps}.}

\item{maxit_comp}{maximum number of iterations.}

\item{eps_quant}{required numerical error for quantile computation.}

\item{maxit_quant}{maximum number of iterations before stopping the quantile computation.}

\item{lambdas_tol}{maximum value admitted for the weight skewness for both the numerator and the denominator. When it is not NULL (default), elements of lambdas such that the ratio max(lambdas)/lambdas is greater than the specified value are removed.}
}
\value{
The function returns an object of the class \code{MellinQF_ratio} that contains information on the Mellin transform
of the ratio of two linear combinations of positively weighted chi-square random variables. This information can be used in order to
evaluate the density, cumulative distribution and quantile functions of the desired quadratic form.

An object of the class \code{MellinQF_ratio} has the following components:
\itemize{
\item \code{range_q}: the range of quantiles that contains the specified mass of probability \code{rho} in which it
is possible to compute density and CDF preserving the error level \code{eps}.
\item \code{Mellin}: a list containing the values of the Mellin transform (\code{Mellin}),
the corresponding evaluation points (\code{z}), the integration step \code{delta} and the lowest numerator weight (\code{lambda_min}).
\item the inputs \code{rho}, \code{lambdas_num}, \code{lambdas_den}, \code{etas_num}, \code{etas_den}, \code{eps} needed for CDF, PDF and quantile function computation.
}
}
\description{
The function computes the Mellin transform of the ratio of independent and positive definite quadratic forms producing a \code{MellinQF_ratio} object.
The output can be used to evaluate the density, cumulative and quantile functions of the target quadratic form.
}
\details{
The Mellin transform of the ratio of two independent quadratic forms having positive weights \code{lambdas_num} and \code{lambdas_den}
and non-centrality parameters \code{etas_num} and \code{etas_den} is computed
exploiting the density formulation by Ruben (1962).  The numerical error is controlled in order to provide the requested precision (\code{eps}) for the
interval of quantiles that contains the specified total probability \code{rho}.

The argument \code{eps_quant} controls the relative precision requested for the computation of quantiles that determine the range in which the error \code{eps} is
guaranteed, whereas \code{maxit_quant} sets the maximum number of Newton-Raphson iterations of the algorithm.
}
\examples{

\donttest{
library(QF)
# Definition of the QFs
lambdas_QF_num <- c(rep(7, 6),rep(3, 2))
etas_QF_num <- c(rep(6, 6), rep(2, 2))
lambdas_QF_den <- c(0.6, 0.3, 0.1)
# Computation Mellin transform
eps <- 1e-7
rho <- 0.999
Mellin_ratio <- compute_MellinQF_ratio(lambdas_QF_num, lambdas_QF_den,
                                       etas_QF_num, eps = eps, rho = rho)
print(Mellin_ratio)
}



}
\seealso{
The function \code{\link{print.MellinQF_ratio}} can be used to summarize the basic information on the Mellin transform.

The object can be used in the function \code{\link{dQF_ratio}} to compute the density function of the QFs ratio,
\code{\link{pQF_ratio}} for the CDF and \code{\link{qQF_ratio}} for the quantile function.
}
