\name{plot_boxtraj}

\alias{plot_boxtraj}

\title{Visualization of Peeling Trajectories/Profiles}

\description{
  Function for plotting the cross-validated peeling trajectories/profiles of a \code{PRSP} object.
  Applies to the pre-selected covariates specified by user and 
  all other statistical quantities of interest at each iteration of the peeling sequence 
  (inner loop of our PRSP algorithm). 
}

\usage{
  plot_boxtraj(object,
               main = NULL, 
               xlab = "Box Mass", 
               ylab = "Covariate Range",
               toplot = object$cvfit$cv.used,
               col.cov, 
               lty.cov, 
               lwd.cov,
               col = 1, 
               lty = 1, 
               lwd = 1, 
               cex = 1, 
               add.legend = FALSE, 
               text.legend = NULL, 
               nr = NULL, 
               nc = NULL,
               device = NULL, 
               file = "Covariate Trajectory Plots", 
               path=getwd(), 
               horizontal = FALSE, 
               width = 8.5, 
               height = 11.5, ...)
}

\arguments{
  \item{object}{
    Object of class \code{PRSP} as generated by the main function \code{\link[PRIMsrc]{sbh}}.
  }
  \item{main}{
    \code{Character} \code{vector}. Main Title. Defaults to \code{NULL}.
  }
  \item{xlab}{
    \code{Character} \code{vector}. X axis label. Defaults to "Box Mass".
  }
  \item{ylab}{
    \code{Character} \code{vector}. Y axis label. Defaults to "Covariate Range".
  }
  \item{toplot}{
    \code{Numeric} \code{vector}. Which of the pre-selected covariates to plot (in reference to the original index of covariates).
    Defaults to covariates used for peeling.
  }
  \item{col.cov}{
    \code{Integer} \code{vector}. Line color for the covariate trajectory curve of each selected covariate. 
    Defaults to vector of colors of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{lty.cov}{
    \code{Integer} \code{vector}. Line type for the covariate trajectory curve of each selected covariate. 
    Defaults to vector of 1's of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{lwd.cov}{
    \code{Integer} \code{vector}. Line width for the covariate trajectory curve of each selected covariate. 
    Defaults to vector of 1's of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{col}{
    \code{Integer} scalar. Line color for the trajectory curve of each statistical quantity of interest. 
    Defaults to 1.
  }
  \item{lty}{
    \code{Integer} scalar. Line type for the trajectory curve of each statistical quantity of interest. 
    Defaults to 1.
  }
  \item{lwd}{
    \code{Integer} scalar. Line width for the trajectory curve of each statistical quantity of interest. 
    Defaults to 1.
  }
  \item{cex}{
    \code{Integer} scalar. Symbol expansion used for titles, legends, and axis labels. Defaults to 1.
  }
  \item{add.legend}{
    \code{Logical} scalar. Should the legend be added to the current open graphics device? Defaults to \code{FALSE}.
  }
  \item{text.legend}{
    \code{Character} \code{vector} of legend content. Defaults to \code{NULL}.
  }
  \item{nr}{
    \code{Integer} scalar of the number of rows in the plot. If \code{NULL}, defaults to 3.
  }
  \item{nc}{
    \code{Integer} scalar of the number of columns in the plot. If \code{NULL}, defaults to 3.
  }
  \item{device}{
    Graphic display device in \{NULL, "PS", "PDF"\}. Defaults to NULL (standard output screen).
    Currently implemented graphic display devices are "PS" (Postscript) or "PDF" (Portable Document Format).
  }
  \item{file}{
    File name for output graphic. Defaults to "Covariate Trajectory Plots".
  }
  \item{path}{
    Absolute path (without final (back)slash separator). Defaults to working directory path.
  }
  \item{horizontal}{
    \code{Logical} scalar. Orientation of the printed image. Defaults to \code{FALSE}, that is potrait orientation.
  }
  \item{width}{
    \code{Numeric} scalar. Width of the graphics region in inches. Defaults to 8.5.
  }
  \item{height}{
    \code{Numeric} scalar. Height of the graphics region in inches. Defaults to 8.5.
  }
  \item{\dots}{
    Generic arguments passed to other plotting functions.
  }
}

\details{
  The plot limits the display of trajectories to those only covariates that are used for peeling.
  
  The plot includes box descriptive statistics (such as support), survival endpoint statistics
  (such as Maximum Event-Free Time (MEFT), Minimum Event-Free Probability (MEVP), LHR, LRT)
  and prediction performance (such as CER).
}

\value{
  Invisible. None. Displays the plot(s) on the specified \code{device}.
}

\references{
    \itemize{
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2015).
              "\emph{Cross-validation and Peeling Strategies for Survival Bump Hunting using Recursive Peeling Methods.}"
              (Submitted).
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2014).
              "\emph{Cross-Validation of Survival Bump Hunting by Recursive Peeling Methods.}"
              In JSM Proceedings, Survival Methods for Risk Estimation/Prediction Section. Boston, MA, USA.
              American Statistical Association IMS - JSM, p. 3366-3380.
        \item Dazard J-E. and J. S. Rao (2010).
              "\emph{Local Sparse Bump Hunting.}"
              J. Comp Graph. Statistics, 19(4):900-92.
    }
}

\author{
    \itemize{
        \item "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}
        \item "Michael Choe, M.D." \email{mjc206@case.edu}
        \item "Michael LeBlanc, Ph.D." \email{mleblanc@fhcrc.org}
        \item "Alberto Santana, MBA." \email{ahs4@case.edu}
    }
    Maintainer: "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}

    Acknowledgments: This project was partially funded by the National Institutes of Health
    NIH - National Cancer Institute (R01-CA160593) to J-E. Dazard and J.S. Rao.
}

\note{
  End-user plotting function.
}

\keyword{Exploratory Survival/Risk Analysis}
\keyword{Survival/Risk Estimation & Prediction}
\keyword{Non-Parametric Method}
\keyword{Cross-Validation}
\keyword{Bump Hunting}
\keyword{Rule-Induction Method}

\examples{
#===================================================
# Loading the library and its dependencies
#===================================================
library("PRIMsrc")

#=================================================================================
# Simulated dataset #1 (n=250, p=3)
# Replicated Combined Cross-Validation (RCCV)
# Peeling criterion = LRT
# Optimization criterion = LRT
#=================================================================================
CVCOMBREP.synt1 <- sbh(dataset = Synthetic.1, 
                       cvtype = "combined", cvcriterion = "lrt",
                       B = 1, K = 5, 
                       vs = TRUE, cpv = FALSE, 
                       decimals = 2, probval = 0.5, 
                       arg = "beta=0.05,
                              alpha=0.1,
                              minn=10,
                              L=NULL,
                              peelcriterion=\"lr\"",
                       parallel = FALSE, conf = NULL, seed = 123)

plot_boxtraj(object = CVCOMBREP.synt1,
             main = paste("RCCV peeling trajectories for model #1", sep=""),
             xlab = "Box Mass", ylab = "Covariate Range",
             toplot = CVCOMBREP.synt1$cvfit$cv.used,
             device = NULL, file = "Covariate Trajectory Plots", path=getwd(),
             horizontal = FALSE, width = 8.5, height = 8.5)

}
