\name{plot_boxtrace}

\alias{plot_boxtrace}

\title{Visualization of Covariates Traces}

\description{
  S3 generic function for plotting the cross-validated covariates traces of a \code{PRSP} object.
  Plot the cross-validated modal trace curves of covariate importance and covariate usage of the 
  user-specified covariates among the pre-selected ones at each iteration of the peeling sequence 
  (inner loop of our PRSP algorithm). 
}

\usage{
  plot_boxtrace(x,
                main = NULL,
                xlab = "Box Mass", 
                ylab = "Covariate Range (centered)",
                toplot = x$used,
                center = TRUE, 
                scale = FALSE, 
                col.cov, 
                lty.cov, 
                lwd.cov, 
                col = 1, 
                lty = 1, 
                lwd = 1, 
                cex = 1, 
                add.legend = FALSE, 
                text.legend = NULL,
                device = NULL, 
                file = "Covariate Trace Plots", 
                path=getwd(), 
                horizontal = FALSE, 
                width = 8.5, 
                height = 8.5, ...)
}

\arguments{
  \item{x}{
    Object of class \code{PRSP} as generated by the main function \code{\link[PRIMsrc]{sbh}}.
  }
  \item{main}{
    \code{Character} \code{vector}. Main Title. Defaults to.
  }
  \item{xlab}{
    \code{Character} \code{vector}. X axis label. Defaults to "Box Mass".
  }\code{NULL}
  \item{ylab}{
    \code{Character} \code{vector}. Y axis label. Defaults to "Covariate Range (centered)".
  }
  \item{toplot}{
    \code{Numeric} \code{vector}. Which of the pre-selected covariates to plot.
    Defaults to covariates used for peeling.
  }
  \item{center}{
    \code{Logical} scalar. Shall the data be centered?. Defaults to \code{TRUE}.
  }
  \item{scale}{
    \code{Logical} scalar. Shall the data be scaled? Defaults to \code{FALSE}.
  }
  \item{col.cov}{
    \code{Integer} \code{vector}. Line color for the covariate importance curve of each selected covariate. 
    Defaults to vector of colors of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{lty.cov}{
    \code{Integer} \code{vector}. Line type for the covariate importance curve of each selected covariate. 
    Defaults to vector of 1's of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{lwd.cov}{
    \code{Integer} \code{vector}. Line width for the covariate importance curve of each selected covariate. 
    Defaults to vector of 1's of length the number of selected covariates. 
    The vector is reused cyclically if it is shorter than the number of selected covariates.
  }
  \item{col}{
    \code{Integer} scalar. Line color for the covariate trace curve. 
    Defaults to 1.
  }
  \item{lty}{
    \code{Integer} scalar. Line type for the covariate trace curve. 
    Defaults to 1.
  }
  \item{lwd}{
    \code{Integer} scalar. Line width for the covariate trace curve. 
    Defaults to 1.
  }
  \item{cex}{
    \code{Integer} scalar. Symbol expansion used for titles, legends, and axis labels. Defaults to 1.
  }
  \item{add.legend}{
    \code{Logical} scalar. Should the legend be added to the current open graphics device?. Defaults to \code{FALSE}.
  }
  \item{text.legend}{
    \code{Character} \code{vector} of legend content. Defaults to \code{NULL}.
  }
  \item{device}{
    Graphic display device in \{NULL, "PS", "PDF"\}. Defaults to NULL (standard output screen).
    Currently implemented graphic display devices are "PS" (Postscript) or "PDF" (Portable Document Format).
  }
  \item{file}{
    File name for output graphic. Defaults to "Covariate Trace Plots".
  }
  \item{path}{
    Absolute path (without final (back)slash separator). Defaults to working directory path.
  }
  \item{horizontal}{
    \code{Logical} scalar. Orientation of the printed image. Defaults to \code{FALSE}, that is potrait orientation.
  }
  \item{width}{
    \code{Numeric} scalar. Width of the graphics region in inches. Defaults to 8.5.
  }
  \item{height}{
    \code{Numeric} scalar. Height of the graphics region in inches. Defaults to 8.5.
  }
  \item{\dots}{
    Generic arguments passed to other plotting functions.
  }
}

\details{
  The trace plots limit the display of traces to those only covariates that are used for peeling. 
  If centered, an horizontal black dotted line about 0 is added to the plot.
 
  Due to the variability induced by cross-validation and replication, it is possible that more than one covariate be used for peeling at a given step.
  So, for simplicity of the trace plots, only the modal or majority vote trace value (over the folds and replications of the cross-validation) is plotted. 
  
  The top plot shows the overlay of covariate importance curves for each covariate.
  The bottom plot shows the overlay of covariate usage curves for each covariate. It is a dicretized view of covariate importance.
  
  Both point to the magnitude and order with which covariates are used along the peeling sequence.
}

\value{
  Invisible. None. Displays the plot(s) on the specified \code{device}.
}

\references{
    \itemize{
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2015).
              "\emph{Cross-validation and Peeling Strategies for Survival Bump Hunting using Recursive Peeling Methods.}"
              (Submitted).
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2014).
              "\emph{Cross-Validation of Survival Bump Hunting by Recursive Peeling Methods.}"
              In JSM Proceedings, Survival Methods for Risk Estimation/Prediction Section. Boston, MA, USA.
              American Statistical Association IMS - JSM, p. 3366-3380.
        \item Dazard J-E. and J. S. Rao (2010).
              "\emph{Local Sparse Bump Hunting.}"
              J. Comp Graph. Statistics, 19(4):900-92.
    }
}

\author{
    \itemize{
        \item "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}
        \item "Michael Choe, M.D." \email{mjc206@case.edu}
        \item "Michael LeBlanc, Ph.D." \email{mleblanc@fhcrc.org}
        \item "Alberto Santana, MBA." \email{ahs4@case.edu}
    }
    Maintainer: "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}

    Acknowledgments: This project was partially funded by the National Institutes of Health
    NIH - National Cancer Institute (R01-CA160593) to J-E. Dazard and J.S. Rao.
}

\note{
  End-user plotting function.
}

\keyword{Exploratory Survival/Risk Analysis}
\keyword{Survival/Risk Estimation & Prediction}
\keyword{Non-Parametric Method}
\keyword{Cross-Validation}
\keyword{Bump Hunting}
\keyword{Rule-Induction Method}

\examples{
#===================================================
# Loading the library and its dependencies
#===================================================
library("PRIMsrc")

#=================================================================================
# Simulated dataset #1 (n=250, p=3)
# Replicated Combined Cross-Validation (RCCV)
# Peeling criterion = LRT
# Optimization criterion = LRT
#=================================================================================
CVCOMBREP.synt1 <- sbh(dataset = Synthetic.1, 
                       cvtype = "combined", cvcriterion = "lrt",
                       B = 1, K = 5, cpv = FALSE, probval = 0.5, 
                       arg = "beta=0.05,
                              alpha=0.1,
                              minn=10,
                              L=NULL,
                              peelcriterion=\"lr\"",
                       parallel = FALSE, conf = NULL, seed = 123)

plot_boxtrace(x = CVCOMBREP.synt1,
              main = paste("RCCV trace plots for model #1", sep=""),
              xlab = "Box Mass", ylab = "Covariate Range (centered)",
              toplot = CVCOMBREP.synt1$used,
              center = TRUE, scale = FALSE,
              device = NULL, file = "Covariate Trace Plots", path=getwd(),
              horizontal = FALSE, width = 8.5, height = 8.5)
}

