\name{distanceHartmann}
\alias{distanceHartmann}
\title{'Hartmann distance' (standardized Slater distances).}
\usage{
  distanceHartmann(x, method = "paper", reps = 10000,
    prob = NULL, progress = TRUE, distributions = FALSE)
}
\arguments{
  \item{x}{\code{repgrid} object.}

  \item{method}{The method used for distance calculation,
  on of \code{"paper", "simulate", "new"}. \code{"paper"}
  uses the parameters as given in Hartmann (1992) for
  caclulation.  \code{"simulate"} (default) simulates a
  Slater distribution for the caclulation. In a future
  version the time consuming simulation will be replaced by
  more accurate parameters for Hartmann distances than used
  in Hartmann (1992).}

  \item{reps}{Number of random grids to generate sample
  distribution for Slater distances (default is
  \code{10000}). Note that a lot of samples may take a
  while to calculate.}

  \item{prob}{The probability of each rating value to
  occur.  If \code{NULL} (default) the distribution is
  uniform.  The number of values must match the length of
  the rating scale.}

  \item{progress}{Whether to show a progress bar during
  simulation (default is \code{TRUE}) (for
  \code{method="simulate"}).  May be useful when the
  distribution is estimated on the basis of many quasis.}

  \item{distributions}{Wether to additionally return the
  values of the simulated distributions (Slater etc.) The
  default is \code{FALSE} as it will quickly boost the
  object size.}
}
\value{
  A matrix containing Hartmann distances. \cr In the
  attributes several additional parameters can be found:
  \cr \item{\code{"arguments"}}{A list of several
  parameters including \code{mean} and \code{sd} of Slater
  distribution.} \item{\code{"quantiles"}}{Quantiles for
  Slater and Hartmann distance distribition.}
  \item{\code{"distributions"}}{List with values of the
  simulated distributions.}
}
\description{
  Calculate Hartmann distance
}
\details{
  Hartmann (1992) showed in a simulation study that Slater
  distances (see \code{\link{distanceSlater}}) based on
  random grids, for which Slater coined the expression
  quasis, have a skewed distribution, a mean and a standard
  deviation depending on the number of constructs elicited.
  He suggested a linear transformation (z-transformation)
  which takes into account the estimated (or expected) mean
  and the standard deviation of the derived distribution to
  standardize Slater distance scores across different grid
  sizes. 'Hartmann distances' represent a more accurate
  version of 'Slater distances'. Note that Hartmann
  distances are multiplied by -1. Hence, negative Hartmann
  values represent dissimilarity, i.e. a big Slater
  distance. \cr

  There are two ways to use this function. Hartmann
  distances can either be calculated based on the reference
  values (i.e. means and standard deviations of Slater
  distance distributions) as given by Hartmann in his
  paper. The second option is to conduct an instant
  simulation for the supplied grid size for each
  calculation. The second option will be more accurate when
  a big number of quasis is used in the simulation. \cr

  It is also possible to return the quantiles of the sample
  distribution and only the element distances consideres
  'significant' according to the quantiles defined.
}
\section{Calculation}{
  The 'Hartmann distance' is calculated as follows
  (Hartmann 1992, p. 49).  \cr \deqn{D = -1
  (\frac{D_{slater} - M_c}{sd_c})}{D = -1 (D_slater - M_c /
  sd_c)} Where \eqn{D_{slater}}{D_slater} denotes the
  Slater distances of the grid, \eqn{M_c}{M_c} the sample
  distribution's mean value and \eqn{sd_c}{sd_c} the sample
  distributions's standard deviation.
}
\examples{
\dontrun{

   ### basics  ###

   distanceHartmann(bell2010)
   distanceHartmann(bell2010, method="simulate")
   h <- distanceHartmann(bell2010, method="simulate")
   h

   # printing options
   print(h)
   print(h, digits=6)
   # 'significant' distances only
   print(h, p=c(.05, .95))

   # access cells of distance matrix
   h[1,2]

   ### advanced  ###

   # histogram of Slater distances and indifference region
   h <- distanceHartmann(bell2010, distributions=TRUE)
   l <- attr(h, "distributions")
   hist(l$slater, breaks=100)
   hist(l$hartmann, breaks=100)
}
}
\author{
  Mark Heckmann
}
\references{
  Hartmann, A. (1992). Element comparisons in repertory
  grid technique: Results and consequences of a Monte Carlo
  study. \emph{International Journal of Personal Construct
  Psychology, 5}(1), 41-56.
}
\seealso{
  \code{\link{distanceSlater}}
}

