% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Olink_anova.R
\name{olink_anova_posthoc}
\alias{olink_anova_posthoc}
\title{Function which performs an ANOVA posthoc test per protein.}
\usage{
olink_anova_posthoc(
  df,
  olinkid_list = NULL,
  variable,
  covariates = NULL,
  outcome = "NPX",
  model_formula,
  effect,
  effect_formula,
  mean_return = FALSE,
  post_hoc_padjust_method = "tukey",
  verbose = TRUE
)
}
\arguments{
\item{df}{NPX data frame in long format with at least protein name (Assay), OlinkID, UniProt, Panel and a factor with at least 3 levels.}

\item{olinkid_list}{Character vector of OlinkID's on which to perform post hoc analysis. If not specified, all assays in df are used.}

\item{variable}{Single character value or character array.
Variable(s) to test. If length > 1, the included variable names will be used in crossed analyses .
Also takes ':' notation.}

\item{covariates}{Single character value or character array. Default: NULL. Covariates to include. Takes ':' or '*' notation. Crossed analysis will not be inferred from main effects.}

\item{outcome}{Character. The dependent variable. Default: NPX.}

\item{model_formula}{(optional) Symbolic description of the model to be fitted in standard formula notation (e.g. "NPX~A*B"). If provided, this will override the \code{outcome}, \code{variable} and \code{covariates} arguments. Can be a string or of class \code{stats::formula()}.}

\item{effect}{Term on which to perform post-hoc. Character vector. Must be subset of or identical to variable.}

\item{effect_formula}{(optional) A character vector specifying the names of the predictors over which estimated marginal means are desired as defined in the \code{emmeans} package. May also be a formula. If provided, this will override the \code{effect} argument. See \code{?emmeans::emmeans()} for more information.}

\item{mean_return}{Boolean. If true, returns the mean of each factor level rather than the difference in means (default). Note that no p-value is returned for mean_return = TRUE and no adjustment is performed.}

\item{post_hoc_padjust_method}{P-value adjustment method to use for post-hoc comparisons within an assay. Options include \code{tukey}, \code{sidak}, \code{bonferroni} and \code{none}.}

\item{verbose}{Boolean. Default: True. If information about removed samples, factor conversion and final model formula is to be printed to the console.}
}
\value{
A "tibble" of posthoc tests for specified effect, arranged by ascending adjusted p-values.
Columns include:
\itemize{
 \item{Assay:} "character" Protein symbol
 \item{OlinkID:} "character" Olink specific ID
 \item{UniProt:} "character" Olink specific ID
 \item{Panel:} "character" Name of Olink Panel
 \item{term:} "character" term in model
 \item{contrast:} "character" the groups that were compared
 \item{estimate:} "numeric" difference in mean NPX between groups
 \item{conf.low:} "numeric" confidence interval for the mean (lower end)
 \item{conf.high:} "numeric" confidence interval for the mean (upper end)
 \item{Adjusted_pval:} "numeric" adjusted p-value for the test
 \item{Threshold:} "character" if adjusted p-value is significant or not (< 0.05)
}
}
\description{
Performs a post hoc ANOVA test using emmeans::emmeans with Tukey p-value adjustment per assay (by OlinkID) for each panel at confidence level 0.95.
See \code{olink_anova} for details of input notation. \cr\cr
The function handles both factor and numerical variables and/or covariates.
The posthoc test for a numerical variable compares the difference in means of the outcome variable (default: NPX) for 1 standard deviation difference in the numerical variable, e.g.
mean NPX at mean(numerical variable) versus mean NPX at mean(numerical variable) + 1*SD(numerical variable).
}
\examples{
\donttest{

library(dplyr)

npx_df <- npx_data1 \%>\% filter(!grepl('control',SampleID, ignore.case = TRUE))

#Two-way ANOVA, one main effect (Site) covariate.
#Results in model NPX~Treatment*Time+Site.
anova_results <- olink_anova(df = npx_df,
                             variable=c("Treatment:Time"),
                             covariates="Site")

#Posthoc test for the model NPX~Treatment*Time+Site,
#on the interaction effect Treatment:Time with covariate Site.

#Filtering out significant and relevant results.
significant_assays <- anova_results \%>\%
filter(Threshold == 'Significant' & term == 'Treatment:Time') \%>\%
select(OlinkID) \%>\%
distinct() \%>\%
pull()

#Posthoc, all pairwise comparisons
anova_posthoc_results <- olink_anova_posthoc(npx_df,
variable=c("Treatment:Time"),
covariates="Site",
olinkid_list = significant_assays,
effect = "Treatment:Time")


#Posthoc, treated vs untreated at each timepoint, adjusted for Site effect
anova_posthoc_results <- olink_anova_posthoc(npx_df,
model_formula = "NPX~Treatment*Time+Site",
olinkid_list = significant_assays,
effect_formula = "pairwise~Treatment|Time")


}
}
