# Close a connection to an OpenOffice.org database (odb) file
# Author : Sylvain Mareschal
odb.close = function(
		odb,
		write = TRUE
		)
	{
	# Class checks
	if (!is(odb, "odb")) {
		stop("'odb' must be an 'odb' object")
	}
	validObject(odb)
	
	# Updates the .odb file properly
	if (write) {
		# HSQLDB disconnection with file compaction
		tryCatch(
			dbSendUpdate(odb, "SHUTDOWN COMPACT"),
			error = function(e) {
				stop("HSQL disconnection failed : \"", conditionMessage(e), "\"")
			}
		)
		
		# Renames HSQLDB files back
		dbFiles = c("backup", "data", "properties", "script")
		dbFiles.odb = paste(odb@directory, "/database/", dbFiles, sep="")
		dbFiles.jdbc = paste(odb@directory, "/database/ODB.", dbFiles, sep="")
		for(i in 1:length(dbFiles)) {
			if (file.exists(dbFiles.jdbc[i])) {
				file.rename(dbFiles.jdbc[i], dbFiles.odb[i])
			}
		}
		
		# Updates the ODB archive
		odb.zip(
			archive = odb@odbFile,
			files = c("content.xml", "database"),
			from = odb@directory,
			recursive = TRUE,
			quiet = TRUE
		)
	} else {
		# HSQLDB disconnection attempt
		tryCatch(
			dbSendUpdate(odb, "SHUTDOWN"),
			error = function(e) {
				warning("HSQL disconnection failed : \"", conditionMessage(e), "\"")
			}
		)
	}
	
	# DBI disconnection attempt
	error = tryCatch(
		dbDisconnect(odb),
		error = function(e) {
			warning("DBI disconnection failed : \"", conditionMessage(error), "\"")
		}
	)
	
	# Removes temporary files
	unlink(odb@directory, recursive=TRUE)
	
	invisible(TRUE)
}
