% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/NlcOptim.R
\name{NlcOptim}
\alias{NlcOptim}
\title{Solve Optimization problem with Nonlinear Objective and Constraints}
\usage{
NlcOptim(X = NULL, objfun = NULL, confun = NULL, A = NULL, B = NULL,
  Aeq = NULL, Beq = NULL, lb = NULL, ub = NULL, tolX = 1e-05,
  tolFun = 1e-06, tolCon = 1e-06, maxnFun = 1e+07, maxIter = 4000)
}
\arguments{
\item{X}{Starting vector of parameter values.}

\item{objfun}{Nonlinear objective function that is to be optimized.}

\item{confun}{Nonlinear constraint function. Return a \code{ceq} vector
and a \code{c} vector as nonlinear equality constraints and an inequality constraints.}

\item{A}{A in the linear inequality constraints.}

\item{B}{B in the linear inequality constraints.}

\item{Aeq}{Aeq in the linear equality constraints.}

\item{Beq}{Beq in the linear equality constraints.}

\item{lb}{Lower bounds of parameters.}

\item{ub}{Upper bounds of parameters.}

\item{tolX}{The tolerance in X.}

\item{tolFun}{The tolerance in the objective function.}

\item{tolCon}{The tolenrance in the constraint function.}

\item{maxnFun}{Maximum updates in the objective function.}

\item{maxIter}{Maximum iteration.}
}
\value{
Return a list with the following components:
\item{p}{The optimum solution.}
\item{fval}{The value of the objective function at the optimal point.}
\item{lambda}{Lagrangian multiplier.}
\item{grad}{The gradient of the objective function at the optimal point.}
\item{hessian}{Hessian of the objective function at the optimal point.}
}
\description{
Sequential Quatratic
Programming (SQP) method is implemented to find solution for general nonlinear optimization problem
(with nonlinear objective and constraint functions). The SQP method can be find in detail in Chapter 18 of
Jorge Nocedal and Stephen J. Wright's book.
Linear or nonlinear equality and inequality constraints are allowed.
It accepts the input parameters as a constrained matrix.
The function \code{NlcOptim} is to solve generalized nonlinear optimization problem:
\deqn{min f(x)}
\deqn{s.t. ceq(x)=0}
\deqn{c(x)\le 0}
\deqn{Ax\le B}
\deqn{Aeq x \le Beq}
\deqn{lb\le x \le ub}
}
\examples{
library(MASS)
###ex1
objfun=function(x){
 return(exp(x[1]*x[2]*x[3]*x[4]*x[5]))
}
#constraint function
confun=function(x){
 f=NULL
 f=rbind(f,x[1]^2+x[2]^2+x[3]^2+x[4]^2+x[5]^2-10)
 f=rbind(f,x[2]*x[3]-5*x[4]*x[5])
 f=rbind(f,x[1]^3+x[2]^3+1)
 return(list(ceq=f,c=NULL))
}

x0=c(-2,2,2,-1,-1)
NlcOptim(x0,objfun=objfun,confun=confun)

####ex2
obj=function(x){
 return((x[1]-1)^2+(x[1]-x[2])^2+(x[2]-x[3])^3+(x[3]-x[4])^4+(x[4]-x[5])^4)
}
#constraint function
con=function(x){
 f=NULL
 f=rbind(f,x[1]+x[2]^2+x[3]^3-2-3*sqrt(2))
 f=rbind(f,x[2]-x[3]^2+x[4]+2-2*sqrt(2))
 f=rbind(f,x[1]*x[5]-2)
 return(list(ceq=f,c=NULL))
}

x0=c(1,1,1,1,1)
NlcOptim(x0,objfun=obj,confun=con)


##########ex3
obj=function(x){
 return((1-x[1])^2+(x[2]-x[1]^2)^2)
}
#constraint function
con=function(x){
 f=NULL
 f=rbind(f,x[1]^2+x[2]^2-1.5)
 return(list(ceq=NULL,c=f))
}

x0=as.matrix(c(-1.9,2))
objfun(x0)
confun(x0)
NlcOptim(x0,objfun=obj,confun=con)


##########ex4
objfun=function(x){
 return(x[1]^2+x[2]^2)
}
#constraint function
confun=function(x){
 f=NULL
 f=rbind(f,-x[1] - x[2] + 1)
 f=rbind(f,-x[1]^2 - x[2]^2 + 1)
 f=rbind(f,-9*x[1]^2 - x[2]^2 + 9)
 f=rbind(f,-x[1]^2 + x[2])
 f=rbind(f,-x[2]^2 + x[1])
 return(list(ceq=NULL,c=f))
}

x0=as.matrix(c(3,1))
NlcOptim(x0,objfun=objfun,confun=confun)
}
\author{
Xianyan Chen, Xiangrong Yin
}
\references{
Nocedal, Jorge, and Stephen Wright. Numerical optimization. Springer Science & Business Media, 2006.
}

