#' Sensitivity analysis plot over time of the data
#'
#' @description Plot of sensitivity of the neural network output respect
#' to the inputs over the time variable from the data provided
#' @param object \code{SensMLP} object generated by \code{\link[NeuralSens]{SensAnalysisMLP}}
#' with several outputs (classification MLP)
#' @param comb_type Function to combine the matrixes of the \code{raw_sens} component of \code{object}.
#' It can be "mean", "median" or "sqmean". It can also be a function to combine the rows of the matrixes
#' @return \code{SensMLP} object with the sensitivities combined
#' @examples
#' \dontrun{
#' # mod should be a neural network classification model
#' sens <- SensAnalysisMLP(mod)
#' combinesens <- CombineSens(sens, "sqmean")
#' }
#' @export CombineSens
CombineSens <- function(object, comb_type = "mean") {
  # Check that is a valid object
  if (!is.SensMLP(object)) stop("object is not a recognized SensMLP object")
  if (!"sens" %in% names(object)) stop("sens not found in object")
  if (!"raw_sens" %in% names(object)) stop("raw_sens not found in object")

  # Check that there are several sensitivities to combine
  if (length(object$sens) == 1 || length(object$raw_sens) == 1) {
    warning("Only found 1 sensitivity to combine, return the same object")
    return(object)
  }

  # Combine sensitivity measures
  sens_to_return <- object$sens[[1]]
  means <- NULL
  stds <- NULL
  sqmeans <- NULL
  for (i in 1:length(object$sens)) {
    means[[i]] <- object$sens[[i]]$mean
    stds[[i]] <- object$sens[[i]]$std
    sqmeans[[i]] <- object$sens[[i]]$meanSensSQ
  }
  means <- as.data.frame(means)
  stds <- as.data.frame(stds)
  sqmeans <- as.data.frame(sqmeans)
  sens_to_return$mean <- rowMeans(means, na.rm = TRUE)
  # sens_to_return$std <- sqrt(rowMeans(apply(stds, 2, function(x){x^2})))
  # Formula to combine the stds extracted from
  # https://www.researchgate.net/post/How_to_combine_standard_deviations_for_three_groups
  # and assuming that n >> 1 (reasonable if we have trained a neural network)
  sens_to_return$std <- sqrt(rowMeans(apply(stds, 2, function(x){x^2})) +
                               rowMeans(apply(means, 2, function(x){(x - sens_to_return$mean)^2})))
  sens_to_return$meanSensSQ <- rowMeans(sqrt(sqmeans), na.rm = TRUE) ^ 2
  object$sens <- list(Combined = sens_to_return)

  raw_sens <- array(NA,dim = c(dim(object$raw_sens[[1]]),length(object$raw_sens)))
  for (i in 1:length(object$raw_sens)) {
    raw_sens[,,i] <- object$raw_sens[[i]]
  }
  # Only accept mean and mean square
  if(is.function(comb_type)) {
    raw_sens <- apply(raw_sens, c(1,2), comb_type)
  } else if (comb_type == "mean") {
    raw_sens <- apply(raw_sens, c(1,2), mean, na.rm = TRUE)
  } else if (comb_type == "median") {
    raw_sens <- apply(raw_sens, c(1,2), stats::median, na.rm = TRUE)
  } else if (comb_type == "sqmean") {
    raw_sens <- apply(raw_sens, c(1,2), function(x){mean(x^2, na.rm = TRUE)})
  } else {
    stop("comb_type must be a function to combine rows")
  }
  object$raw_sens <- list(Combined = raw_sens)
  return(object)
}
