##' Read simulation results based on NMsim's track of model runs
##' @param x Path to the simulation-specific rds file generated by
##'     NMsim, typically called `NMsim_paths.rds`. Can also be a table
##'     of simulation runs as stored in `rds` files by `NMsim`. The
##'     latter should almost never be used.
##' @param wait If simulations seem to not be done yet, wait for them
##'     to finish? If not, an error will be thrown. If you choose to
##'     wait, the risk is results never come. `NMreadSim` will be
##'     waiting for an `lst` file. If Nonmem fails, it will normally
##'     generate an `lst` file. But if `NMTRAN` fails (checks of
##'     control stream prior to running Nonmem), the `lst` file is not
##'     generated.  Default is not to wait.
##' @param quiet Turn off some messages about what is going on?
##'     Default is to report the messages.
##' @param check.time If found, check whether `fst` file modification
##'     time is newer than `rds` file. The `fst` is generated based on
##'     information in `rds`, but notice that some systems don't
##'     preserve the file modification times. Becasue of that,
##'     `check.time` is `FALSE` by default.
##' @param dir.sims By default, `NMreadSim` will use information about
##'     the relative path from the results table file (`_paths.rds`)
##'     to the Nonmem simulation results. If these paths have changed,
##'     or for other reasons this doesn't work, you can use the
##'     `dir.sims` argument to specify where to find the Nonmem
##'     simulation results. If an `.fst` file was already generated
##'     and is found next to the `_paths.rds`, the path to the Nonmem
##'     simulation results is not used.
##' @param as.fun The default is to return data as a data.frame. Pass
##'     a function (say `tibble::as_tibble`) in as.fun to convert to
##'     something else. If data.tables are wanted, use
##'     as.fun="data.table". The default can be configured using
##'     NMdataConf.
##' @return A data set of class defined by as.fun
##' @import NMdata
##' @import data.table
##' @import fst
##' @importFrom xfun relative_path
##' @export



NMreadSim <- function(x,check.time=FALSE,dir.sims,wait=FALSE,quiet=FALSE,as.fun){

#### Section start: Dummy variables, only not to get NOTE's in pacakge checks ####

    . <- NULL
    path.sim.lst <- NULL
    pathResFromSims <- NULL
    is.fst <- NULL
    is.lst <- NULL
    is.rds <- NULL
    is.res <- NULL
    is.simRes <- NULL
    is.simModTab <- NULL
    is.ModTab <- NULL
    is.ROWEL <- NULL
    ROWEL <- NULL

### Section end: Dummy variables, only not to get NOTE's in pacakge checks

    if(missing(dir.sims)) dir.sims <- NULL
    if(missing(as.fun)) as.fun <- NULL
    as.fun <- NMdata:::NMdataDecideOption("as.fun",as.fun)

    if(is.data.frame(x)) x <- list(x)
### recognized formats:
    ## NMsimRes - return x

    ## NMSimModels - read results in tab

    ## path to rds - read rds, then fst or lst
    

    if(length(x)==0) {
        message("No elementes in x. Returning NULL.")
        return(NULL)
    }

    
    
    dt.x <- data.table(is.rds=unlist(lapply(x,function(x)is.character(x)&&fnExtension(x)=="rds")))
    dt.x[,is.fst:=unlist(lapply(x,function(x)is.character(x)&&fnExtension(x)=="fst"))]
    dt.x[,is.simRes:=unlist(lapply(x,is.NMsimRes))]
    dt.x[,is.simModTab:=unlist(lapply(x,is.NMsimModTab))]
    dt.x[,ROWEL:=.I]

    dt.x[,is.res:=is.fst|is.simRes]
    dt.x[,is.ModTab:=is.rds|is.simModTab]

    if(dt.x[is.res==TRUE&is.ModTab==TRUE,.N]>0) stop("confused, an object seems to be both a NMsimModTab and an NMsimRes")
    if(dt.x[is.res!=TRUE&is.ModTab!=TRUE,.N]>0) {

        ## some may try to read lsts because of behavior of previos versions. Checking explicitly for those.
        dt.x[,is.lst:=unlist(lapply(x,function(x)is.character(x)&&fnExtension(x)=="lst"))]
        if(any(dt.x$is.lst)){
            stop("`lst` (Nonmem output control stream) files found in `x`. NMreadSim() does not read those, and it is not recommended to read simulation results from those directly. Please use `rds` files generated by `NMsim()` because they contain more information. If you need to read `lst` files from earlier NMsim versions, please use NMdata::NMscanData() or NMdata::NMscanMultiple().")
        }
        
        stop("Not all objects in `x` recognized by NMreadSim. They should be either (normally) paths to `rds` files or (mostly for programming) tables of simulation model information.")

        }

    
    
    res.all <- NULL
    if(sum(dt.x$is.simRes)){
        res.simRes <- NMreadSimRes(x[dt.x$is.simRes])
        if(is.null(res.all)){
            res.all <- res.simRes
        } else {
            res.all <- rbind(res.all,res.simRes)  
        }
    }

    if(sum(dt.x$is.ModTab)){
        res.modTab <- NMreadSimModTab(x[dt.x$is.ModTab],check.time=check.time,
                                      dir.sims=dir.sims,wait=wait,quiet=quiet)
        if(is.null(res.all)){
            res.all <- res.modTab
        } else {
            res.all <- rbind(res.all,res.modTab)  
        }
    }
    
    ##    res.all <- rbind(res.simRes,res.modTab,fill=TRUE)

    res.all <- as.fun(res.all)
    addClass(res.all,"NMsimRes")
    return(res.all)
}




