% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothSpline.R
\name{smooth.spline.raster}
\alias{smooth.spline.raster}
\title{Filter Time Series Imagery with a Cubic Spline}
\usage{
smooth.spline.raster(
  x,
  w = NULL,
  t = NULL,
  groupYears = TRUE,
  timeInfo = orgTime(x),
  df = 6,
  outDirPath = "./",
  ...
)
}
\arguments{
\item{x}{\code{RasterBrick} (or \code{RasterStack}) or \code{character} vector of file
names, sorted 'Vegetation index'.}

\item{w}{\code{RasterBrick} (or \code{RasterStack}) with weighting information, e.g.
derived from \code{\link[=makeWeights]{makeWeights()}}.}

\item{t}{In case of MODIS composite, the corresponding
'composite_day_of_the_year' \code{RasterBrick} (or \code{RasterStack}).}

\item{groupYears}{\code{logical}. If \code{TRUE} (default), output files are grouped by
years.}

\item{timeInfo}{Result from \code{\link[=orgTime]{orgTime()}}.}

\item{df}{\code{numeric}, yearly degree of freedom value passed to
\code{\link[stats:smooth.spline]{stats::smooth.spline()}}. If set as \code{character} (i.e., \code{df = "6"}), it is
not adapted to the time series length but used as a fixed value (see
Details).}

\item{outDirPath}{Output path, defaults to the current working directory.}

\item{...}{Arguments passed to \code{\link[raster:writeRaster]{raster::writeRaster()}}. Note that 'filename'
is created automatically.}
}
\value{
The filtered data and a text file with the dates of the output layers.
}
\description{
This function uses the \code{\link[stats:smooth.spline]{stats::smooth.spline()}} function to filter a
vegetation index time series of satellite data.
}
\details{
\code{numeric} values of 'df' (e.g., \code{df = 6}) are treated as yearly degrees of
freedom. Here, the length of the input time series is not relevant since \code{df}
is adapted to it with: \code{df * ('length of _input_ timeserie in days' / 365)}.
The input length can differ from the output because of the 'pillow' argument
in \code{\link[=orgTime]{orgTime()}}.

\code{character} values of 'df' (e.g., \code{df = "6"}), on the other hand, are not
adopted to the length of the input time series.

Currently tested on MODIS and Landsat data. With M*D13 data, it is also
possible to use the 'composite_day_of_the_year' layer and the 'VI_Quality'
layer.
}
\examples{
\dontrun{
# The full capacity of the following functions is currently available only 
# with M*D13 data.
# !! The function is very new, double check the result!!

# You need to extract the: 'vegetation index', 'VI_Quality layer', 
# and 'composite day of the year' layer.
# runGdal(product="MOD13A2",begin="2004340",extent="sicily",end="2006070",
# job="fullCapa",SDSstring="101000000010")
# Afterward extract it to: 
options("MODIS_outDirPath")

# the only obligatory dataset is "x" (vegetatino index), get the 'vi' data on
# the source directory: 
path <- paste0(options("MODIS_outDirPath"),"/fullCapa")
vi <- preStack(path=path, pattern="*_NDVI.tif$")

# `orgTime()` detects timing information of the input data and generates 
# based on the arguments the output date information. For spline functions 
# (in general) the start and end of the time series is always problematic. 
# So there is the argument "pillow" (default 75 days) that adds
# (if available) some more layers on the two endings.

timeInfo <- orgTime(vi,nDays=16,begin="2005001",end="2005365",pillow=40)

# now re-run "preStack" with two differences, 'files' (output of the first 
# `preStack()` call) and the 'timeInfo'.
# Here only the data needed for the filtering is extracted:
vi <- preStack(files=vi,timeInfo=timeInfo)

smooth.spline.raster(x=vi,timeInfo=timeInfo)

# Filter with weighting and time information:
# if the files are M*D13 you can use also use quality layers and the 
# composite day of the year:
w <- stack(preStack(path=path, pattern="*_VI_Quality.tif$", timeInfo=timeInfo))
w <- makeWeights(w,bitShift=2,bitMask=15,threshold=6)
# you can also pass only the names
t <- preStack(path=path, pattern="*_composite_day_of_the_year.tif$", timeInfo=timeInfo)

smooth.spline.raster(x=vi,w=w,t=t,timeInfo=timeInfo)
}

}
\seealso{
\code{\link[=whittaker.raster]{whittaker.raster()}}, \code{\link[raster:raster]{raster::raster()}}.
}
\author{
Matteo Mattiuzzi
}
