#' Extraction d'un bilan qualité
#'
#' Permet d'extraire un bilan qualité à partir du fichier CSV contenant la
#' matrice des diagnostics.
#'
#' @param matrix_output_file fichier CSV contenant la matrice des diagnostics.
#' @param sep séparateur de caractères utilisé dans le fichier csv (par défaut
#' \code{sep = ";"})
#' @param dec séparateur décimal utilisé dans le fichier csv (par défaut
#' \code{dec = ","})
#'
#' @details La fonction permet d'extraire un bilan qualité à partir d'un
#' fichier csv contenant l'ensemble des
#' diagnostics (généralement fichier \emph{demetra_m.csv}).
#'
#' Ce fichier peut être obtenu en lançant le cruncher
#' (\code{\link[rjwsacruncher]{cruncher}} ou
#' \code{\link[rjwsacruncher]{cruncher_and_param}}) avec l'ensemble des
#' paramètres de base pour les paramètres à exporter et l'option
#' \code{csv_layout = "vtable"} (par défaut) pour le format de sortie des
#' fichiers csv (option de \code{\link[rjwsacruncher]{cruncher_and_param}} ou de
#' \code{\link[rjwsacruncher]{create_param_file}} lors de la création du fichier
#' de paramètres).
#'
#' Le résultat de cette fonction est un objet \code{\link{QR_matrix}} qui est
#' une liste de trois paramètres :
#' * le paramètre \code{modalities} est un \code{data.frame} contenant un
#'   ensemble de variables sous forme catégorielle (Good, Uncertain, Bad,
#'   Severe).
#' * le paramètre \code{values} est un \code{data.frame} contenant les valeurs
#'   associées aux indicateurs présents dans \code{modalities} (i.e. :
#'   p-valeurs, statistiques, etc.) ainsi que des variables qui n'ont pas de
#'   modalité (fréquence de la série et modèle ARIMA).
#' * le paramètre \code{score_formula} est initié à \code{NULL} : il contiendra
#'   la formule utilisée pour calculer le score (si le calcul est fait).
#'
#' @encoding UTF-8
#' @return Un objet de type \code{\link{QR_matrix}}.
#' @examples
#' # Chemin menant au fichier demetra_m.csv
#' demetra_path <- file.path(
#'     system.file("extdata", package = "JDCruncheR"),
#'     "WS/ws_ipi/Output/SAProcessing-1",
#'     "demetra_m.csv"
#' )
#'
#' # Extraire le bilan qualité à partir du fichier demetra_m.csv
#' QR <- extract_QR(demetra_path)
#'
#' print(QR)
#'
#' # Extraire les modalités de la matrice
#' QR$modalities
#' # Or:
#' QR[["modalities"]]
#'
#' @keywords internal
#' @name fr-extract_QR
NULL
#> NULL


#' Extraction of a quality report
#'
#' To extract a quality report from the csv file containing the diagnostics
#' matrix.
#'
#' @param matrix_output_file the csv file containing the diagnostics matrix.
#' @param sep the separator used in the csv file (by default, \code{sep = ";"})
#' @param dec the decimal separator used in the csv file (by default,
#' \code{dec = ","})
#'
#' @details This function generates a quality report from a csv file containing
#' diagnostics (usually from the file \emph{demetra_m.csv}).
#' The \emph{demetra_m.csv} file can be generated by launching the cruncher
#' (functions \code{\link[rjwsacruncher]{cruncher}} or
#' \code{\link[rjwsacruncher]{cruncher_and_param}}) with the default export
#' parameters, having used the default option \code{csv_layout = "vtable"} to
#' format the output tables of the functions
#' \code{\link[rjwsacruncher]{cruncher_and_param}} and
#' \code{\link[rjwsacruncher]{create_param_file}} when creating the parameters
#' file.
#'
#' This function returns a \code{\link{QR_matrix}} object, which is a list of 3
#' objects:
#' * \code{modalities}, a \code{data.frame} containing several indicators and
#'   their categorical quality (Good, Uncertain, Bad, Severe).
#' * \code{values}, a \code{data.frame} containing the same indicators and the
#'   values that lead to their quality category (i.e.: p-values, statistics,
#'   etc.) as well as additional variables that don't have a modality/quality
#'   (series frequency and arima model).
#' * \code{score_formula} that will store the formula used to calculate the
#'   score (when relevant). Its initial value is \code{NULL}.
#'
#' @encoding UTF-8
#'
#' @return a \code{\link{QR_matrix}} object.
#'
#' @family QR_matrix functions
#' @examples
#' # Path of matrix demetra_m
#' demetra_path <- file.path(
#'     system.file("extdata", package = "JDCruncheR"),
#'     "WS/ws_ipi/Output/SAProcessing-1",
#'     "demetra_m.csv"
#' )
#'
#' # Extract the quality report from the demetra_m file
#' QR <- extract_QR(demetra_path)
#'
#' print(QR)
#'
#' # Extract the modalities matrix:
#' QR$modalities
#' # Or:
#' QR[["modalities"]]
#'
#' @importFrom stats sd
#' @importFrom utils read.csv
#' @seealso [Traduction française][fr-extract_QR()]
#' @export
extract_QR <- function(matrix_output_file, sep = ";", dec = ",") {
    if (missing(matrix_output_file) || is.null(matrix_output_file)) {
        stop("Please call extract_QR() on a csv file containing at least one cruncher output matrix (demetra_m.csv for example)")
    }
    if (length(matrix_output_file) == 0
        || !file.exists(matrix_output_file)
        || length(grep("\\.csv$", matrix_output_file)) == 0) {
        stop("The chosen file desn't exist or isn't a csv file")
    }

    demetra_m <- read.csv(
        file = matrix_output_file,
        sep = sep,
        dec = dec,
        stringsAsFactors = FALSE,
        na.strings = c("NA", "?"),
        fileEncoding = "latin1",
        quote = ""
    )

    if (nrow(demetra_m) == 0 || ncol(demetra_m) == 0) {
        stop("The chosen csv file is empty")
    }

    tests_variables <- c(
        "qs.test.on.sa", "f.test.on.sa..seasonal.dummies.",
        "qs.test.on.i", "f.test.on.i..seasonal.dummies.",
        "f.test.on.sa..td.", "f.test.on.i..td.", "independence", "normality"
    )
    missing_variables <- setdiff(tests_variables, colnames(demetra_m))
    present_variables <- intersect(tests_variables, colnames(demetra_m))
    index_present_variables <- match(present_variables, tests_variables)
    if (length(missing_variables) != 0) {
        warning(
            "The following variables are missing from the diagnostics matrix:\n",
            paste(missing_variables, collapse = ", "),
            "\n\nIf you are using a workspace with JDemetra+ in v2, please re-compute the export.",
            "\nIf you are using a workspace with JDemetra+ in v3, ignore this warning."
        )
    }

    demetra_m$series <- gsub(
        "(^ *)|(* $)", "",
        gsub("(^.* \\* )|(\\[frozen\\])", "", demetra_m[, 1])
    )
    demetra_m$frequency <- extractFrequency(demetra_m)

    demetra_m <- cbind(
        demetra_m,
        extractARIMA(demetra_m),
        extractStatQ(demetra_m),
        extractOOS_test(demetra_m),
        extractNormalityTests(demetra_m),
        extractOutliers(demetra_m)
    )

    values_name <- unlist(sapply(X = present_variables, FUN = paste0, "_pvalue"))
    colnames(demetra_m)[match(present_variables, colnames(demetra_m)) + 1] <- values_name

    modalities_variables <- c(
        "series",
        present_variables,
        "homoskedasticity_modality", "skewness_modality", "kurtosis_modality", "oos_mean_modality",
        "oos_mse_modality", "m7_modality", "q_modality", "q_m2_modality", "pct_outliers_modality"
    )

    values_variables <- c(
        "series",
        values_name,
        "homoskedasticity_pvalue", "skewness_pvalue", "kurtosis_pvalue", "oos_mean_pvalue",
        "oos_mse_pvalue", "m7", "q_value", "q_m2_value", "pct_outliers_value",
        "frequency", "arima_model"
    )

    if (!all(
        modalities_variables %in% colnames(demetra_m),
        values_variables %in% colnames(demetra_m)
    )) {
        missing_variables <- unique(c(
            modalities_variables[!modalities_variables %in% colnames(demetra_m)],
            values_variables[!values_variables %in% colnames(demetra_m)]
        ))
        missing_variables <- paste(missing_variables, collapse = "\n")
        stop(paste0(
            "The following variables are missing from the diagnostics matrix:\n",
            missing_variables, "\nPlease re-compute the export."
        ))
    }

    names_QR_variables <- c(
        "series",

        c("qs_residual_sa_on_sa", "f_residual_sa_on_sa",
          "qs_residual_sa_on_i", "f_residual_sa_on_i",
          "f_residual_td_on_sa", "f_residual_td_on_i",
          "residuals_independency", "residuals_normality")[index_present_variables],

        "residuals_homoskedasticity", "residuals_skewness", "residuals_kurtosis",
        "oos_mean", "oos_mse", "m7", "q", "q_m2", "pct_outliers"
    )
    QR_modalities <- demetra_m[, modalities_variables]
    QR_values <- demetra_m[, values_variables]
    rownames(QR_modalities) <- rownames(QR_values) <- NULL
    colnames(QR_values)[seq_along(names_QR_variables)] <- colnames(QR_modalities) <- names_QR_variables
    QR_modalities[, -1] <- lapply(
        X = QR_modalities[, -1],
        FUN = factor,
        levels = c("Good", "Uncertain", "Bad", "Severe"),
        ordered = TRUE
    )
    QR <- QR_matrix(modalities = QR_modalities, values = QR_values)
    QR
}

extractFrequency <- function(demetra_m) {
    if (anyNA(match(c("start", "end", "n"), colnames(demetra_m)))) {
        stop("Error in the extraction of the series frequency (missing either the start date, the end date or the number of observations)")
    }
    start <- as.Date(demetra_m$start, format = "%Y-%m-%d")
    end <- as.Date(demetra_m$end, format = "%Y-%m-%d")
    n <- demetra_m$n

    start <- data.frame(y = as.numeric(format(start, "%Y")), m = as.numeric(format(start, "%m")))
    end <- data.frame(y = as.numeric(format(end, "%Y")), m = as.numeric(format(end, "%m")))
    freq <- c(12, 6, 4, 3, 2)
    nobs_compute <- matrix(sapply(freq, function(x) {
        x * (end[, 1] - start[, 1]) + (end[, 2] - start[, 2]) / (12 / x)
    }), nrow = nrow(demetra_m))
    return(sapply(seq_len(nrow(nobs_compute)), function(i) {
        freq[which((nobs_compute[i, ] == n[i]) | (nobs_compute[i, ] + 1 == n[i]) | (nobs_compute[i, ] - 1 == n[i]))[1]]
    }))
}

extractARIMA <- function(demetra_m) {
    q_possibles <- grep("(^q$)|(^q\\.\\d$)", colnames(demetra_m))
    bp_possibles <- grep("(^bp$)|(^bp\\.\\d$)", colnames(demetra_m))

    if (length(q_possibles) > 1) {
        val_q <- demetra_m[, q_possibles]
        integer_col <- which(sapply(val_q, is.integer))
        if (length(integer_col) == 0) {
            val_q <- rep(NA_integer_, nrow(demetra_m))
        } else if (length(integer_col) == 1) {
            val_q <- val_q[, integer_col[1]]
        }
    } else if (length(q_possibles) == 1) {
        val_q <- demetra_m[, q_possibles]
    } else  {
        stop("Error in the extraction of the arima order q: multiple column.")
    }

    if (length(bp_possibles) > 1) {
        val_bp <- demetra_m[, bp_possibles]
        integer_col <- which(sapply(val_bp, is.integer))
        if (length(integer_col) == 0) {
            val_bp <- rep(NA_integer_, nrow(demetra_m))
        } else {
            val_bp <- val_bp[, integer_col[1]]
        }
    } else if (length(bp_possibles) == 1) {
        val_bp <- demetra_m[, bp_possibles]
    } else  {
        stop("Error in the extraction of the arima order bp: multiple column.")
    }

    if (!all(
        is.integer(val_q) || all(is.na(val_q)),
        is.integer(val_bp) || all(is.na(val_bp))
    )) {
        stop("Error in the extraction of the arima order q or bp")
    }
    arima <- data.frame(
        arima_p = demetra_m[, "p"],
        arima_d = demetra_m[, "d"],
        arima_q = val_q,
        arima_bp = val_bp,
        arima_bd = demetra_m[, "bd"],
        arima_bq = demetra_m[, "bq"]
    )
    arima$arima_model <- paste0(
        "(", arima$arima_p, ",", arima$arima_d, ",", arima$arima_q, ")",
        "(", arima$arima_bp, ",", arima$arima_bd, ",", arima$arima_bq, ")"
    )
    return(arima)
}

extractStatQ <- function(demetra_m, thresholds = getOption("jdc_thresholds")) {

    col_q <- q_possibles <- grep("(^q$)|(^q\\.\\d$)",
                                 colnames(demetra_m))
    col_q_m2 <- q_m2_possibles <- grep("(^q\\.m2$)|(^q\\.m2\\.\\d$)",
                                       colnames(demetra_m))

    if (length(q_possibles) > 1) {
        col_q_possibles <- demetra_m[, q_possibles]
        NA_cols <- which(unlist(lapply(
            X = col_q_possibles,
            FUN = function(x) all(is.na(x))
        )))
        num_cols <- which(unlist(lapply(
            X = col_q_possibles,
            FUN = function(x) !all(is.integer(x) | is.character(x) | is.na(x))
        )))

        if (length(num_cols) > 1) {
            stop("Error in the extraction of the Q stats: multiple colum found")
        } else if (length(num_cols) == 1) {
            col_q <- q_possibles[num_cols]
        } else if (length(NA_cols) > 0) {
            col_q <- q_possibles[NA_cols[1]]
        } else {
            stop("Error in the extraction of the Q stats")
        }
    }

    if (length(q_m2_possibles) > 1) {
        col_q_m2_possibles <- demetra_m[, q_m2_possibles]
        NA_cols <- which(unlist(lapply(
            X = col_q_m2_possibles,
            FUN = function(x) all(is.na(x))
        )))
        num_cols <- which(unlist(lapply(
            X = col_q_m2_possibles,
            FUN = function(x) !all(is.integer(x) | is.character(x) | is.na(x))
        )))

        if (length(num_cols) > 1) {
            stop("Error in the extraction of the Q stats: multiple colum found")
        } else if (length(num_cols) == 1) {
            col_q_m2 <- q_m2_possibles[num_cols]
        } else if (length(NA_cols) > 0) {
            col_q_m2 <- q_m2_possibles[NA_cols[1]]
        } else {
            stop("Error in the extraction of the Q-M2 stats")
        }
    }

    stat_Q <- data.frame(
        q_modality = cut(
            x = as.numeric(demetra_m[, col_q]),
            breaks = c(-Inf, thresholds[["q"]]),
            labels = names(thresholds[["q"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        q_value = demetra_m[, col_q],
        q_m2_modality = cut(
            x = as.numeric(demetra_m[, col_q_m2]),
            breaks = c(-Inf, thresholds[["q_m2"]]),
            labels = names(thresholds[["q_m2"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        q_m2_value = demetra_m[, col_q_m2],
        stringsAsFactors = FALSE
    )

    return(stat_Q)
}

extractOOS_test <- function(demetra_m, thresholds = getOption("jdc_thresholds")) {

    col_mean <- mean_possibles <- grep("(^mean$)|(^mean\\.\\d$)", colnames(demetra_m))

    if (length(mean_possibles) > 1) {
        col_mean_possibles <- demetra_m[, mean_possibles]
        NA_cols <- which(unlist(lapply(
            X = col_mean_possibles,
            FUN = function(x) all(is.na(x))
        )))
        num_cols <- which(unlist(lapply(
            X = col_mean_possibles,
            FUN = function(x) !all(is.integer(x) | is.character(x) | is.na(x))
        )))

        if (length(num_cols) > 1) {
            stop("Error in the extraction of the mean in the out of sample diagnostics: multiple column")
        } else if (length(num_cols) == 1) {
            col_mean <- mean_possibles[num_cols]
        } else if (length(NA_cols) > 0) {
            col_mean <- mean_possibles[NA_cols[1]]
        } else {
            stop("Error in the extraction of the mean in the out of sample diagnostics")
        }

    }

    col_mse <- match("mse", colnames(demetra_m))
    if (all(is.na(col_mse))) {
        val_mse <- rep(NA_real_, nrow(demetra_m))
    } else if (length(col_mse) == 1L) {
        val_mse <- demetra_m[, col_mse]
        if (is.character(val_mse)) {
            col_mse <- col_mse + 1
            val_mse <- demetra_m[, col_mse]
        }
    } else {
        stop("Error in the extraction of the mse in the out of sample diagnostics: multiple column")
    }

    stat_OOS <- data.frame(
        oos_mean_modality = cut(
            x = as.numeric(demetra_m[, col_mean]),
            breaks = c(-Inf, thresholds[["oos_mean"]]),
            labels = names(thresholds[["oos_mean"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        oos_mean_pvalue = as.numeric(demetra_m[, col_mean]),
        oos_mse_modality = cut(
            x = as.numeric(val_mse),
            breaks = c(-Inf, thresholds[["oos_mse"]]),
            labels = names(thresholds[["oos_mse"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        oos_mse_pvalue = as.numeric(val_mse),
        stringsAsFactors = FALSE
    )
    return(stat_OOS)
}

extractNormalityTests <- function(demetra_m, thresholds = getOption("jdc_thresholds")) {
    tests_possibles <- grep("(^skewness$)|(^kurtosis$)|(^lb2$)", colnames(demetra_m))
    if (length(tests_possibles) != 3) {
        stop("At least one test is missing, among: skewness, kurtosis, lb2")
    }

    if (is.character(demetra_m$skewness)
        && is.character(demetra_m$kurtosis)
        && is.character(demetra_m$lb2)) {
        if (length(grep(pattern = "^X\\.(\\d){1,}$", x = colnames(demetra_m)[rep(tests_possibles, each = 2) + rep(1:2, 3)])) != 6) {
            stop("Re-compute the cruncher export with the options: residuals.skewness:3, residuals.kurtosis:3 and residuals.lb2:3")
        }
        skewness_pvalue <- demetra_m[, tests_possibles[1] + 2]
        kurtosis_pvalue <- demetra_m[, tests_possibles[2] + 2]
        homoskedasticity_pvalue <- demetra_m[, tests_possibles[3] + 2]

    } else if (is.numeric(demetra_m$skewness)
               && is.numeric(demetra_m$kurtosis)
               && is.numeric(demetra_m$lb2)) {
        if (length(grep(pattern = "^X\\.(\\d){1,}$", x = colnames(demetra_m)[tests_possibles + 1])) != 3) {
            stop("Re-compute the cruncher export with the options: residuals.skewness:2, residuals.kurtosis:2 and residuals.lb2:2")
        }
        skewness_pvalue <- demetra_m[, tests_possibles[1] + 1]
        kurtosis_pvalue <- demetra_m[, tests_possibles[2] + 1]
        homoskedasticity_pvalue <- demetra_m[, tests_possibles[3] + 1]
    } else {
        stop("the matrix has wrong format.")
    }

    normality <- data.frame(
        skewness_pvalue = skewness_pvalue,
        kurtosis_pvalue = kurtosis_pvalue,
        homoskedasticity_pvalue = homoskedasticity_pvalue,
        skewness_modality = cut(
            x = skewness_pvalue,
            breaks = c(-Inf, thresholds[["residuals_skewness"]]),
            labels = names(thresholds[["residuals_skewness"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        kurtosis_modality = cut(
            x = skewness_pvalue,
            breaks = c(-Inf, thresholds[["residuals_kurtosis"]]),
            labels = names(thresholds[["residuals_kurtosis"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        ),
        homoskedasticity_modality = cut(
            x = skewness_pvalue,
            breaks = c(-Inf, thresholds[["residuals_normality"]]),
            labels = names(thresholds[["residuals_normality"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        )
    )
    return(normality)
}

extractOutliers <- function(demetra_m, thresholds = getOption("jdc_thresholds")) {

    col_nbr_outliers <- match("number.of.outliers", colnames(demetra_m))
    if (all(is.na(col_nbr_outliers))) {
        pct_outliers_modality <- rep(NA_character_, nrow(demetra_m))
        pct_outliers_value <- rep(NA_real_, nrow(demetra_m))
    } else if (length(col_nbr_outliers) == 1) {
        pct_outliers_modality <- demetra_m[, col_nbr_outliers]
        pct_outliers_value <- demetra_m[, col_nbr_outliers + 1] * 100
    } else if (length(col_nbr_outliers) > 1) {
        stop("Error in the extraction of the number of outliers: multiple column")
    }

    outliers <- data.frame(
        pct_outliers_modality = pct_outliers_modality,
        pct_outliers_value = pct_outliers_value,
        m7_modality = cut(
            x = as.numeric(demetra_m$m7),
            breaks = c(-Inf, thresholds[["m7"]]),
            labels = names(thresholds[["m7"]]),
            right = FALSE,
            include.lowest = TRUE,
            ordered_result = TRUE
        )
    )

    return(outliers)
}
