#' Display marker names
#'
#' This is a help function to extract marker names from an object generated by
#'  \link{genDataRead}, \link{genDataLoad}, \link{genDataGetPart} or 
#'  \link{genDataPreprocess}.
#'
#' @param data.in The data as outputted by \link{genDataRead}, \link{genDataLoad}, 
#'   \link{genDataGetPart} or \link{genDataPreprocess}.
#' @param n Number of names to display or "all" (default: 5).
#' @param from From which marker to display (optional, default: from the first).
#' @param to To which marker to display (optional).
#'
#' @return A vector with marker names, as read in from map.file or generated dummy names.
#'
showSNPnames <- function( data.in, n = 5, from, to ){
	marker.names <- data.in$aux$marker.names
	nmarkers <- length( marker.names )

	n.given <- FALSE
	from.given <- FALSE
	which.markers.show <- TRUE

	if( !missing( n ) ){
		n.given <- TRUE
		if( is.numeric( n ) ){
			if( n < 1 | n > nmarkers ){
				stop( "Your specification of 'n' (number of names to display) was wrong!", call. = FALSE )
			}
			# n is numeric and in correct range
		} else if( n != "all" ){
			stop( "Could not understand the choice of 'n'", call. = FALSE )
		} else {
			# n == "all"
			n <- nmarkers
		}
	}

	if( !missing( from ) ){
		from.given <- TRUE
		if( !is.numeric( from ) ){
			stop( "'From' should be numeric!", call. = FALSE )
		}
	}

	if( !missing( to ) ){
		if( !is.numeric( to ) ){
			stop( "'To' should be numeric!", call. = FALSE )
		}
		if( n.given & from.given ){
			warning( "You specified too many parameters! Chosing only 'from' and 'to'." )
		} else if( n.given ) {
			# only 'n' and 'to' are specified
			from <- to - n + 1
		} else if( !from.given ) {
			# only 'to' is specified
			from <- 1
		}
	} else if( n.given & from.given ) {
	# only 'from' and 'n' is specified
		to <- from + n - 1
	} else if( from.given ) {
	# only 'from' is specified
		to <- nmarkers
	} else if( n.given ) {
	# only 'n' is specified
		from <- 1
		to <- from + n - 1
	} else {
		# nothing specified: print the first 5 entries
		from <- 1
		to <- min( n, nmarkers )
	}

	if( from < 1 | to > nmarkers | from > to ){
		stop( paste0( "Wrong specification of 'from' (", from ,") and 'to' (", to, ")!" ), call. = FALSE )
	}
	which.markers.show <- from:to

	return( marker.names[ which.markers.show ] )
}
