#' Generate correct text using the Generative AI model based on a prompt
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters and generates correct text based on the prompt.
#'
#' @param model.parameter A character vector containing the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. A higher value results in more creative responses,
#' while a lower value produces more straightforward text.
#' @param prompt A character string representing the query for text generation.
#'
#' @return If successful, a character string (correct text) generated by the Generative AI
#' model based on the provided query and parameters. If the API response indicates an error,
#' the function halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial
#' to ensure successful text generation by the Generative AI model. If any of the
#' provided parameters is incorrect, the function will respond with an error message based
#' on the information received from the API. Use the function \code{available.models} to
#' see all supported Generative AI models.
#'
#' A complete prompt behind the function is as follows:
#'
#' Rewrite the following text and fix any grammar issues:
#'
#' # Text starts #
#'
#' \code{prompt}
#'
#' # Text ends #
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Generate text
#'  temperature = 0.9
#'  prompt = "Yesterday, I will buy a book for my younger sister as his birthday gift.
#'            They were very happen when seeing this gift earlier today."
#'  correct.text = text.fix.grammar(google.model,
#'                                  temperature,
#'                                  prompt)
#'  cat(correct.text)
#'
#'  correct.text = text.fix.grammar(openai.model,
#'                                  temperature,
#'                                  prompt)
#'  cat(correct.text)
#' }
#'
#' @export
#'
#' @importFrom GenAI moderation.openai
text.fix.grammar = function (model.parameter, temperature, prompt) {
  if (prompt == "" || is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            requestBody = list(
              contents = list(parts = list(text = paste("Rewrite the following text and fix any grammar issues:\n ",
                                                        "# Text starts #\n",
                                                        prompt, "\n",
                                                        "# Text ends #\n"))),
              generationConfig = list(temperature = temperature)
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("The prompt may contain harmful content.")
            }
            return (as.character(responseJSON$candidates[[1]]$content$parts[[1]]$text))
          },
          openai = {
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            requestBody = list(
              model = model.parameter["model"],
              messages = list(
                list(role = "system",
                     content = "You are a helpful assistant."),
                list(role = "user",
                     content = paste("Rewrite the following text and fix any grammar issues:\n ",
                                     "# Text starts #\n",
                                     prompt, "\n",
                                     "# Text ends #\n"))
              ),
              temperature = temperature
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            return (as.character(responseJSON$choices[[1]]$message$content))
          })
}
