% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gPoMo.R
\name{gPoMo}
\alias{gPoMo}
\title{Generalized Polynomial Modeling}
\usage{
gPoMo(data, tin = NULL, dtFixe = NULL, dMax = 2, nS = c(3), winL = 9,
  weight = NULL, show = 1, verbose = 1, underSamp = NULL, EqS = NULL,
  IstepMin = 2, IstepMax = 2000, nPmin = 1, nPmax = 14,
  method = "lsoda")
}
\arguments{
\item{data}{Input Time series: Each column is one time series
that corresponds to one variable.}

\item{tin}{Input date vector which length should correspond to
the input time series.}

\item{dtFixe}{Time step used for the analysis. It should correspond
to the sampling time of the input data.
Note that for very large and very small time steps, alternative units
may be used in order to stabilize the numerical computation.}

\item{dMax}{Maximum degree of the polynomial formulation.}

\item{nS}{A vector providing the number of dimensions used for each
input variables (see Examples 1 and 2). The dimension of the resulting
model will be \code{nVar = sum(nS)}.}

\item{winL}{Total number of points used for computing the derivatives
of the input time series. This parameter will be used as an
input in function \code{drvSucc} to compute the derivatives.}

\item{weight}{A vector providing the binary weighting function
of the input data series (0 or 1). By default, all the values
are set to 1.}

\item{show}{Provide (2) or not (0-1) visual output during
the running process.}

\item{verbose}{Gives information (if set to 1) about the algorithm
progress and keeps silent if set to 0.}

\item{underSamp}{Number of points used for undersampling the data.
For \code{undersamp = 1} the complete time series is used.
For \code{undersamp = 2}, only one data out of two is kept, etc.}

\item{EqS}{Model template including all allowed regressors.
Each column corresponds to one equation. Each line corresponds to one
polynomial term as defined by function \code{poLabs}.}

\item{IstepMin}{The minimum number of integration step to start
of the analysis (by default \code{IstepMin = 10}).}

\item{IstepMax}{The maximum number of integration steps for
stopping the analysis (by default \code{IstepMax = 10000}).}

\item{nPmin}{Corresponds to the minimum number of parameters (and thus
of polynomial term) allowed.}

\item{nPmax}{Corresponds to the maximum number of parameters (and thus
of polynomial) allowed.}

\item{method}{The integration technique used for the numerical
integration. By default, the fourth-order Runge-Kutta method
(\code{method = 'rk4'}) is used. Other methods such as 'ode45'
or 'lsoda' may also be chosen. See package \code{deSolve}
for details.}
}
\value{
A list containing:

\code{$tin}        The time vector of the input time series

\code{$inputdata}  The input time series

\code{$tfiltdata}  The time vector of the filtered time series (boudary removed)

\code{$filtdata}   A matrix of the filtered time series with its derivatives

\code{$okMod}      A vector classifying the models: diverging models (0), periodic
models of period-1 (-1), unclassified models (1).

\code{$coeff}      A matrix with the coefficients of one selected model

\code{$models}     A list of all the models to be tested \code{$mToTest1},
\code{$mToTest2}, etc. and all selected models \code{$model1}, \code{$model2}, etc.

\code{$tout}       The time vector of the output time series (vector length
corresponding to the longest numerical integration duration)

\code{$stockoutreg} A list of matrices with the integrated trajectories
(variable \code{X1} in column 1, \code{X2} in 2, etc.) of all the models \code{$model1},
\code{$model2}, etc.
}
\description{
Algorithm for a Generalized Polynomial formulation
of multivariate Global Modeling.
Global modeling aims to obtain multidimensional models
from single time series [1-2].
In the generalized (polynomial) formulation provided in this
function, it can also be applied to multivariate time series [3-4].

Example:\cr
Note that \code{nS} provides the number of dimensions used from each variable

case I \cr
For \code{nS=c(2,3)} means that 2 dimensions are reconstructed from variable 1:
the original variable \code{X1} and its first derivative \code{X2}), and
3 dimensions are reconstructed from variable 2: the original
variable \code{X3} and its first and second derivatives \code{X4} and \code{X5}.
The generalized model will thus be such as: \cr
\eqn{dX1/dt = X2}\cr
\eqn{dX2/dt = P1(X1,X2,X3,X4,X5)}\cr
\eqn{dX3/dt = X4}\cr
\eqn{dX4/dt = X5}\cr
\eqn{dX5/dt = P2(X1,X2,X3,X4,X5).}\cr

case II\cr
For \code{nS=c(1,1,1,1)} means that only the original variables
\code{X1}, \code{X2}, \code{X3} and \code{X4} will be used.
The generalized model will thus be such as:\cr
\eqn{dX1/dt = P1(X1,X2,X3,X4)}\cr
\eqn{dX2/dt = P2(X1,X2,X3,X4)}\cr
\eqn{dX3/dt = P3(X1,X2,X3,X4)}\cr
\eqn{dX4/dt = P4(X1,X2,X3,X4).}\cr
}
\examples{
#Example 1
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,3]
dev.new()
out1 <- gPoMo(data, tin=tin, dMax = 2, nS=c(3), show = 1,
              IstepMax = 1000, nPmin = 9, nPmax = 11)
visuEq(3, 2, out1$models$model1, approx = 4)

\dontrun{
#Example 2
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,2:4]
dev.new()
out2 <- gPoMo(data, tin=tin, dMax = 2, nS=c(1,1,1), show = 1,
              IstepMin = 10, IstepMax = 3000, nPmin = 7, nPmax = 8)
# the simplest model able to reproduce the observed dynamics is model #5
visuEq(3, 2, out2$models$model5, approx = 4) # the original Rossler system is thus retrieved
}

\dontrun{
#Example 3
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,2:3]
# model template:
EqS <- matrix(1, ncol = 3, nrow = 10)
EqS[,1] <- c(0,0,0,1,0,0,0,0,0,0)
EqS[,2] <- c(1,1,0,1,0,1,1,1,1,1)
EqS[,3] <- c(0,1,0,0,0,0,1,1,0,0)
visuEq(3, 2, EqS, substit = c('X','Y','Z'))
dev.new()
out3 <- gPoMo(data, tin=tin, dMax = 2, nS=c(2,1), show = 1,
      EqS = EqS, IstepMin = 10, IstepMax = 2000,
      nPmin = 9, nPmax = 11)
}

\dontrun{
#Example 4
# load data
data("TSallMod_nVar3_dMax2")
#multiple (six) time series
tin <- TSallMod_nVar3_dMax2$SprK$reconstr[1:400,1]
TSRo76 <- TSallMod_nVar3_dMax2$R76$reconstr[,2:4]
TSSprK <- TSallMod_nVar3_dMax2$SprK$reconstr[,2:4]
data <- cbind(TSRo76,TSSprK)[1:400,]
dev.new()
# generalized Polynomial modelling
out4 <- gPoMo(data, tin = tin, dMax = 2, nS = c(1,1,1,1,1,1),
              show = 0, method = 'rk4',
              IstepMin = 2, IstepMax = 3,
              nPmin = 13, nPmax = 13)

# the original Rossler (variables x, y and z) and Sprott (variables u, v and w)
# systems are retrieved:
visuEq(6, 2, out4$models$model347, approx = 4,
       substit = c('x', 'y', 'z', 'u', 'v', 'w'))
# to check the robustness of the model, the integration duration
# should be chosen longer (at least IstepMax = 4000)
}

}
\references{
[1] Gouesbet G. & Letellier C., 1994. Global vector-field reconstruction by using a
multivariate polynomial L2 approximation on nets, Physical Review E, 49 (6),
4955-4972. \cr
[2] Mangiarotti S., Coudret R., Drapeau L. & Jarlan L., Polynomial search and
Global modelling: two algorithms for modeling chaos. Physical Review E, 86(4),
046205. \cr
[3] Mangiarotti S., Le Jean F., Huc M. & Letellier C., Global Modeling of aggregated
and associated chaotic dynamics. Chaos, Solitons and Fractals, 83, 82-96. \cr
[4] S. Mangiarotti, M. Peyre & M. Huc, 2016.
A chaotic model for the epidemic of Ebola virus disease
in West Africa (2013-2016). Chaos, 26, 113112. \cr
}
\seealso{
\code{\link{gloMoId}}, \code{\link{autoGPoMoSearch}},
         \code{\link{autoGPoMoTest}}

\code{\link{autoGPoMoSearch}}, \code{\link{autoGPoMoTest}}, \code{\link{visuOutGP}},
         \code{\link{poLabs}}, \code{\link{predictab}}, \code{\link{drvSucc}}
}
\author{
Sylvain Mangiarotti, Flavie Le Jean, Mireille Huc
}
