% Generated by roxygen2 (4.0.1): do not edit by hand
\name{FOLP.ordFun}
\alias{FOLP.ordFun}
\title{Solves a fuzzy objective linear programming problem using ordering functions.}
\usage{
FOLP.ordFun(objective, A, dir, b, maximum = TRUE, ordf = c("Yager1",
  "Yager3", "Adamo", "Average", "Custom"), ..., FUN = NULL)
}
\arguments{
\item{objective}{A vector \eqn{(c_{1}^{f}, c_{2}^{f}, ..., c_{n}^{f})}{(c1, c2, ..., cn)} of
Trapezoidal Fuzzy Numbers with the objective function coefficients
\eqn{f(x)=c_{1}^{f} x_1+\ldots+c_{n}^{f} x_n}{f(x)=c1*x1+\ldots+cn*xn}. Note that any of the
coefficients may also be Real Numbers.}

\item{A}{Technological matrix of Real Numbers.}

\item{dir}{Vector of strings with the direction of the inequalities, of the same length as \code{b}. Each element
of the vector must be one of "=", ">=", "<=", "<" or ">".}

\item{b}{Vector with the right hand side of the constraints.}

\item{maximum}{\code{TRUE} to maximize the objective function, \code{FALSE} to minimize the objective function.}

\item{ordf}{Ordering function to be used, ordf must be one of "Yager1", "Yager3", "Adamo", "Average" or "Custom".
The "Custom" option allows to use a custom linear ordering function that must be placed as FUN argument.
If a non linear function is used the solution may not be optimal.}

\item{...}{Additional parameters to the ordering function if needed.
\itemize{
  \item Yager1 doesn't need any parameters.
  \item Yager3 doesn't need any parameters.
  \item Adamo needs a \code{alpha} parameter which must be in the interval \code{[0,1]}.
  \item Average needs two parameters, \code{lambda} must be in the interval \code{[0,1]} and
  \code{t} that must be greater than \code{0}.
  \item If Custom function needs parameters, put them here. Although not required, it is recommended
  to name the parameters.
}}

\item{FUN}{Custom linear ordering function to be used if the value of ordf is "Custom". If any of the
coefficients of the objective function are Real Numbers, the user must assure that the function
\code{FUN} works well not only with Trapezoidal Fuzzy Numbers but also with Real Numbers.}
}
\value{
\code{FOLP.ordFun} returns the solution if the solver has found it or NULL if not.
}
\description{
The goal is to solve a linear programming problem having Trapezoidal Fuzzy Numbers
as coefficients in the objective function (\eqn{f(x)=c_{1}^{f} x_1+\ldots+c_{n}^{f} x_n}{f(x)=c1*x1+\ldots+cn*xn}).
\deqn{Max\, f(x)\ or\ Min\ f(x)}{Max f(x) or Min f(x)}
\deqn{s.t.:\quad Ax<=b}{s.t.:  Ax<=b}

\code{FOLP.ordFun} uses ordering functions to compare Fuzzy Numbers.
}
\examples{
## maximize:   [0,2,3]*x1 + [1,3,4,5]*x2
## s.t.:         x1 + 3*x2 <= 6
##               x1 +   x2 <= 4
##               x1, x2 are non-negative real numbers

obj <- c(TrapezoidalFuzzyNumber(0,2,2,3), TrapezoidalFuzzyNumber(1,3,4,5))
A<-matrix(c(1, 1, 3, 1), nrow = 2)
dir <- c("<=", "<=")
b <- c(6, 4)
max <- TRUE

FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Yager1")
FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Yager3")
FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Adamo", 0.5)
FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Average", lambda=0.8, t=3)

# Define a custom linear function
av <- function(tfn) {mean(core(tfn))}
FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Custom", FUN=av)

# Define a custom linear function
avp <- function(tfn, a) {a*mean(core(tfn))}
FOLP.ordFun(obj, A, dir, b, maximum = max, ordf="Custom", FUN=avp, a=2)
}
\references{
Gonzalez, A. A studing of the ranking function approach through mean values. Fuzzy Sets and Systems, 35:29-41, 1990.

Cadenas, J.M. and Verdegay, J.L. Using Fuzzy Numbers in Linear Programming. IEEE Transactions on Systems, Man, and Cybernetics-Part B: Cybernetics, vol. 27, No. 6, December 1997.

Tanaka, H., Ichihashi, H. and Asai, F. A formulation of fuzzy linear programming problems based a comparison of fuzzy numbers. Control and Cybernetics, 13:185-194, 1984.
}
\seealso{
\code{\link{FOLP.multiObj}}, \code{\link{FOLP.interv}}, \code{\link{FOLP.strat}}, \code{\link{FOLP.posib}}
}

