context("comb_CLS")

test_that("Forward wrong input to CLS", {
  expect_error(comb_CLS(1))
  expect_error(comb_CLS("abs"))
  expect_error(comb_CLS(list(a=1, b=2)))
  expect_error(comb_CLS(NULL))
  expect_error(comb_CLS(NA))
  expect_error(comb_CLS(Inf))
  expect_error(comb_CLS(-Inf))
})

test_that("Check for correct class type and accuracy, when only train set is used", {
  set.seed(5)
  obs <- rnorm(100)
  preds <- matrix(rnorm(1000, 1), 100, 10)
  train_o<-obs[1:80]
  train_p<-preds[1:80,]

  data<-foreccomb(train_o, train_p)
  result<-comb_CLS(data)

  expect_is(result, "foreccomb_res")
  expect_length(result, 6)
  expect_equal(as.vector(result$Accuracy_Train), 
               c(-0.9353996, 1.383241, 1.129004, 135.2611, 338.5791, 0.1379442, 0.9836031),
               tolerance = 1e-5, 
               check.attributes = FALSE)
})

test_that( "Check for correct class type and accuracy, when Forecast_Test is provided but not Actual_Test", {
  set.seed(5)
  obs <- rnorm(100)
  preds <- matrix(rnorm(1000, 1), 100, 10)
  train_o<-obs[1:80]
  train_p<-preds[1:80,]
  test_p<-preds[81:100,]
  
  data<-foreccomb(train_o, train_p, newpreds =  test_p)
  result<-comb_CLS(data)
  
  expect_is(result, "foreccomb_res")
  expect_length(result, 7)
  expect_equal(as.vector(result$Accuracy_Train), 
               c(-0.9353996, 1.383241, 1.129004, 135.2611, 338.5791, 0.1379442, 0.9836031),
               tolerance = 1e-5, 
               check.attributes = FALSE)
})

test_that( "Check for correct class type and accuracy, when test set is used", {
  set.seed(5)
  obs <- rnorm(100)
  preds <- matrix(rnorm(1000, 1), 100, 10)
  train_o<-obs[1:80]
  train_p<-preds[1:80,]
  test_o<-obs[81:100]
  test_p<-preds[81:100,]
  
  data<-foreccomb(train_o, train_p, test_o, test_p)
  result<-comb_CLS(data)
  
  expect_is(result, "foreccomb_res")
  expect_length(result, 8)
  expect_equal(as.vector(result$Accuracy_Test), 
               c(-0.9021843, 1.313459, 1.056852, 597.4275, 648.381),
               tolerance = 1e-5, 
               check.attributes = FALSE)
})

test_that( "Check for correct combination, when test set is used with the predict function (simplified)", {
  set.seed(5)
  obs <- rnorm(100)
  preds <- matrix(rnorm(1000, 1), 100, 10)
  train_o<-obs[1:80]
  train_p<-preds[1:80,]
  test_p<-preds[81:100,]
  
  data<-foreccomb(train_o, train_p)
  result<-comb_CLS(data)
  
  data2<-foreccomb(train_o, train_p, newpreds=test_p)
  result2<-comb_CLS(data2)
  
  preds <- predict(result, test_p, simplify = TRUE)
  
  expect_equal(as.vector(preds)[1:5], 
               result2$Forecasts_Test[1:5],
               tolerance = 1e-5, 
               check.attributes = FALSE)
  
})

test_that( "Check for correct combination, when test set is used with the predict function (extend object)", {
  set.seed(5)
  obs <- rnorm(100)
  preds <- matrix(rnorm(1000, 1), 100, 10)
  train_o<-obs[1:80]
  train_p<-preds[1:80,]
  test_p<-preds[81:100,]
  
  data<-foreccomb(train_o, train_p)
  result<-comb_CLS(data)
  
  data2<-foreccomb(train_o, train_p, newpreds=test_p)
  result2<-comb_CLS(data2)
  
  preds <- predict(result, test_p, simplify = FALSE)
  
  expect_equal(as.vector(preds$Forecasts_Test)[1:5], 
               result2$Forecasts_Test[1:5],
               tolerance = 1e-5, 
               check.attributes = FALSE)
})