#' Use the pattern learned from the training data to impute (fill in good guesses for) missing values.
#'
#' Like Amelia, FastImputation assumes that the columns of the data are
#' multivariate normal or can be transformed into approximately
#' multivariate normal.
#' 
#' @param x Vector, matrix, dataframe, or object that can be coerced into a dataframe, possibly with some missing (\code{NA}) values.
#' @param patterns An object of class 'FastImputationPatterns' generated by \code{TrainFastImputation}.
#' @param verbose If TRUE then the progress in imputing the data will be shown.
#' @return An object of class 'FastImputationPatterns' that contains
#'   information needed later to impute on a single row.
#' @export
#' @seealso \code{\link{TrainFastImputation}}
#' @references
#' \url{http://gking.harvard.edu/amelia/}
#' @author Stephen R. Haptonstahl \email{srh@@haptonstahl.org}
#' @examples
#' data(FItrain)   # provides FItrain dataset
#' patterns <- TrainFastImputation(FItrain)
#' 
#' data(FItest)
#' FItest          # note there is missing data
#' imputed.data <- FastImputation(FItest, patterns)
#' imputed.data    # good guesses for missing values are filled in
#'
#' data(FItrue)
#' imputation.rmse <- sqrt(sum( (imputed.data - FItrue)^2 )/sum(is.na(FItest)))
#' imputation.rmse
FastImputation <-
function(
  x,
  patterns,
  verbose=TRUE
) {
  if( missing(patterns) ) {
    stop("A 'patterns' object must be specified.")
  } else {
    if( class(patterns) != "FastImputationPatterns" ) stop("'patterns' must be of class 'FastImputationPatterns'. This is generated by appropriate use of the 'TrainFastImputation' function.")
  }
  if( is.vector(x) && is.numeric(x) ) x <- as.data.frame(t(x))
  if( !is.data.frame(x) && is.matrix(x) ) x <- as.data.frame(x)
  if( !is.data.frame(x) ) stop("'x' must be a vector, matrix, or dataframe.")
  
  n.cols <- length(patterns$FI.means)
  n.rows <- nrow(x)
  
  if(verbose) pb <- txtProgressBar(style=3)
  
  for(i.row in 1:n.rows) {
    # Transform present data if constrained to an interval
    constrained.row <- x[i.row,]
    for(i in patterns$FI.cols.bound.to.intervals) {
      if( !is.na(constrained.row[i]) ) {
        constrained.row[i] <- NormalizeBoundedVariable(
          x=constrained.row[i],
          constraints=patterns$FI.constraints[[i]])
      } 
    }
  
    # Use formula for mean here: http://en.wikipedia.org/wiki/Multivariate_normal_distribution#Conditional_distributions
    cols.to.impute <- which(is.na(constrained.row))    # indices of "1" in Wikipedia formula for mean of conditional multivariate normal distribution
    if( length(cols.to.impute) > 0 ) {
      known.cols <- setdiff(1:n.cols, cols.to.impute)  # incides of "2" in Wikipedia formula for mean of conditional multivariate normal distribution
      
      replacement.values <- t(t(patterns$FI.means[cols.to.impute])) + 
        patterns$FI.covariance[cols.to.impute,known.cols, drop=FALSE] %*% 
        solve(a=patterns$FI.covariance[known.cols,known.cols], 
          b=t(constrained.row[known.cols]) - t(t(patterns$FI.means[known.cols])))
    
      # Store replacement values (note that constraints are not yet applied)
      x[i.row,cols.to.impute] <- replacement.values ### PERHAPS ADD as.vector to RHS
    
      # Constrain interval cols
      imputed.cols.constrained.to.intervals <- intersect(patterns$FI.cols.bound.to.intervals, cols.to.impute)
      for(i in imputed.cols.constrained.to.intervals) {
        x[i.row, i] <- BoundNormalizedVariable(x[i.row, i], constraints=patterns$FI.constraints[[i]])
      }
    
      # Constrain set cols
      imputed.cols.constrained.to.sets <- intersect(patterns$FI.cols.bound.to.sets, cols.to.impute)
      for(i in imputed.cols.constrained.to.sets) {
        x[i.row, i] <- LimitToSet(x[i.row, i], set=patterns$FI.constraints[[i]]$set)
      }
    }
    if(verbose) setTxtProgressBar(pb, i.row/n.rows)
  }
  if(verbose) { 
    close(pb)
    cat("\n")
  }
  return(x)
}
