% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/load.R
\name{farm.test}
\alias{farm.test}
\title{Main function performing factor-adjusted robust test for means}
\usage{
farm.test(X, H0 = NULL, fx = NULL, Kx = NULL, Y = NULL, fy = NULL,
  Ky = NULL, alternative = c("two.sided", "lesser", "greater"),
  alpha = NULL, robust = TRUE, cv = TRUE, tau = 2, verbose = FALSE,
  ...)
}
\arguments{
\item{X}{a n x p data matrix with each row being a sample.
You wish to test a hypothesis for the mean of each column of \code{X}.}

\item{H0}{an \emph{optional} p x 1 vector of the true value of the means (or difference in means if you are performing a two sample test). The default is the zero.}

\item{fx}{an \emph{optional} factor matrix with each column being a factor for \code{X}. Same number of rows as \code{X}.}

\item{Kx}{a \emph{optional} number of factors to be estimated for \code{X}. Otherwise estimated internally. Kx>=0}

\item{Y}{an \emph{optional} data matrix that must have the same number of columns as \code{X}. You wish test the equality of means of each columns of \code{X} and \code{Y}.}

\item{fy}{an \emph{optional} factor matrix with each column being a factor for \code{Y}.  Same number of rows as \code{Y}. Only used for a two sample test.}

\item{Ky}{a \emph{optional} number of factors to be estimated for \code{Y}. Otherwise estimated internally.}

\item{alternative}{an \emph{optional} character string specifying the alternate hypothesis, must be one of "two.sided" (default), "greater" or "lesser". You can specify just the initial letter.}

\item{alpha}{an \emph{optional} level for controlling the false discovery rate (in decimals). Default is 0.05. Must be in \eqn{(0,1)}.}

\item{robust}{a boolean, specifying whether or not to use robust estimators for mean and variance. Default is TRUE.}

\item{cv}{a boolean, specifying whether or  not to run cross-validation for the tuning parameter. Default is TRUE. Only used if \code{robust} is TRUE.}

\item{tau}{\code{>0}, multiplier for the tuning parameter for Huber loss function. Default is 2. Only used if \code{robust} is TRUE and \code{cv} is FALSE. See details.}

\item{verbose}{a boolean specifying whether to print runtime updates to the console. Default is TRUE.}

\item{\dots}{Arguments passed to the \code{\link{farm.FDR}} function.}
}
\value{
An object with S3 class \code{farm.test} containing:
\item{means}{estimated means}
\item{stderr}{estimated standard errors}
\item{pvalue}{unadjusted p values}
\item{rejected}{the indices of rejected hypotheses, along with their corresponding p values, and adjusted p values, ordered from most significant to least significant}
\item{alldata}{all the indices of the tested hypotheses, along with their corresponding p values, adjusted p values, and a column with 1 if declared siginificant and 0 if not}
\item{loadings}{estimated factor loadings}
\item{nfactors}{the number of (estimated) factors}
\item{significant}{the number of means that are found significant}
\item{\dots}{further arguments passed to methods. For complete list use the function \code{\link{names}} on the output object}
}
\description{
This function is used to conduct robust statistical test for means of multivariate data, after adjusting for known or unknown latent factors using the methods in Fan et al.(2017) and Zhou et al.(2017).
It uses the Huber's loss function (Huber (1964)) to robustly estimate data parameters.
}
\details{
\code{alternative = "greater"} is the alternative that \code{X} has a larger mean than \code{Y}.

If some of the underlying factors are known but it is suspected that there are more confounding factors that are unobserved: Suppose we have data \eqn{X = \mu + Bf + Cg + u}, where \eqn{f} is observed and \eqn{g} is unobserved. In the first step, the user passes the data \eqn{\{X,f\}} into the main function. From the output, let us construct the residuals: \eqn{Xres = X - Bf}. Now pass \eqn{Xres} into the main function, without any factors. The output in this step is the final answer to the testing problem.

For two-sample test, the output values \code{means}, \code{stderr}, \code{n}, \code{nfactors},\code{loadings} are all lists containing two items, each pertaining to \code{X} and \code{Y}, indicated by a prefix \code{X.} and \code{Y.} respectively.

Number of rows and columns of the data matrix must be at least 4 in order to be able to calculate latent factors.

For details about multiple comparison correction, see \code{\link{farm.FDR}}.

The tuning parameter \code{= tau *  sigma * optimal rate } where \code{optimal rate } is the optimal rate for the tuning parameter. For details, see Fan et al.(2017). \code{sigma} is the standard deviation of the data.
}
\examples{
set.seed(100)
p = 100
n = 50
epsilon = matrix(rnorm( p*n, 0,1), nrow = n)
B = matrix(runif(p*3,-2,2), nrow=p)
fx = matrix(rnorm(3*n, 0,1), nrow = n)
mu = rep(0, p)
mu[1:5] = 2
X = rep(1,n)\%*\%t(mu)+fx\%*\%t(B)+ epsilon
output = farm.test(X, cv=FALSE)#robust, no cross-validation
output

#other robustification options
output = farm.test(X, robust = FALSE, verbose=FALSE) #non-robust
output = farm.test(X, tau = 3, cv=FALSE, verbose=FALSE) #robust, no cross-validation, specified tau
#output = farm.test(X) #robust, cross-validation, longer running

#two sample test
n2 = 25
epsilon = matrix(rnorm( p*n2, 0,1), nrow = n2)
B = matrix(rnorm(p*3,0,1), nrow=p)
fy = matrix(rnorm(3*n2, 0,1), nrow = n2)
Y = fy\%*\%t(B)+ epsilon
output = farm.test(X=X,Y=Y, robust=FALSE)
output = farm.test(X=X,Y=Y,Kx=0, cv = FALSE) #non-robust
names(output$means)

}
\references{
Huber, P.J. (1964). "Robust Estimation of a Location Parameter." The Annals of Mathematical Statistics, 35, 73–101.

Fan, J., Ke, Y., Sun, Q. and Zhou, W-X. (2017). "FARM-Test: Factor-Adjusted Robust Multiple Testing with False Discovery Control", \url{https://arxiv.org/abs/1711.05386}.

Zhou, W-X., Bose, K., Fan, J. and Liu, H. (2017). "A New Perspective on Robust M-Estimation: Finite Sample Theory and Applications to Dependence-Adjusted Multiple Testing," Annals of Statistics, to appear, \url{https://arxiv.org/abs/1711.05381}.
}
\seealso{
\code{\link{farm.FDR}}, \code{\link{print.farm.test}}
}
