\name{Mqs.uncertainty}
\alias{Mqs.uncertainty}
\title{Function to estimate end-member scores uncertainty}
\description{The function uses either existing assemblages of end-member loadings or 
specified measures of centrality and dispersion as input for Monte 
Carlo runs to estimate the influence of different end-member loadings on
end-member scores. Likewise, the influence of the weight limit quantiles 
(lw) can be estimated.}
\usage{Mqs.uncertainty(X, q, lw, c, rotation = "Varimax", Vqn, Vqn.sd, 
    runs, type.lw, autocorrelation)}
\arguments{
  \item{X}{Numeric matrix with m samples (rows) and n variables (columns).}
  \item{q}{Numeric scalar with the number of end-members to include. Only necessary
in combination with \code{Vqn} as matrix of user-defined end-member
loadings.}
  \item{lw}{Numeric vector with the weight tranformation limits (i.e. quantiles after
Klovan & Imbrie, 1971). If the parameter is of length 1, \code{lw} is assumed to
be a constant, if of length 2, \code{lw} defines either mean and
standard deviation or minimum and maximum, depending on the
value of \code{type.lw}.}
  \item{c}{Numeric scalar specifying the constant sum scaling parameter, e.g. 1, 
100, 1000; default is 100.}
  \item{rotation}{Character scalar, rotation type, default is "Varimax" (cf. Dietze et 
al., 2012). One out of the rotations provided in GPArotation is 
possible (cf. \code{\link{rotations}}).}
  \item{Vqn}{Numeric matrix with existing unscaled end-member loadings. These may 
represent user-defined loadings (or mean loadings if \code{Vqn.sd} is  
specified). See example section for further information.}
  \item{Vqn.sd}{Numeric matrix with standard deviations of the mean unscaled end-member
loadings in \code{Vqn}.}
  \item{runs}{Logical scalar with the number of Monte Carlo runs to be performed,
default is 100.}
  \item{type.lw}{Character scalar with type of random lw value generation. Either
\code{"rnorm"} or \code{"runif"}, default is \code{"runif"}.}
  \item{autocorrelation}{Numeric scalar optionally specifying the degree of autocorrelation among 
classes. Autocorrelation is realised as running mean of the specified 
length. Only odd values are allowed.}
}

\value{A list with numeric vector and matrix objects.
\item{lw}{Randomised weight limit values.}
\item{Vqn}{Randomised unscaled end-member loadings.}
\item{Mqs}{Modelled end-member scores.}
\item{mean}{Modelled end-member score means.}
\item{sd}{Modelled end-member score standard deviations.}}
\references{Dietze E, Hartmann K, Diekmann B, IJmker J, Lehmkuhl F, Opitz S, 
Stauch G, Wuennemann B, Borchers A. 2012. An end-member algorithm for 
deciphering modern detrital processes from lake sediments of Lake Donggi 
Cona, NE Tibetan Plateau, China. Sedimentary Geology 243-244: 169-180.\cr
Klovan JE, Imbrie J. 1971. An Algorithm and FORTRAN-IV Program for 
Large-Scale Q-Mode Factor Analysis and Calculation of Factor Scores. 
Mathematical Geology 3: 61-77.}
\author{Michael Dietze, Elisabeth Dietze}



\seealso{\code{\link{test.robustness}}, \code{\link{test.parameters}}}
\examples{
## load example data set
data(X.artificial, envir = environment())

## set model run parameters
q = 3 # set number of end-members, try 4 to see the difference!
Vqn <- EMMA(X.artificial, q)$Vqn # assign unscaled end-member loadings
Vqn.sd <- Vqn * 0.2 # assign a relative standard deviation of 20 \%
lw.1 <- 0.2 # set lw to 0.2
lw.2 <- c(0.2, 0.08) # set lw to mean = 0.2 and sd = 0.08
runs <-  12 # senseless value to increase comuptation speed

## EXAMPLE 1
## Calculate Mqs uncertainty
M <- Mqs.uncertainty(X = X.artificial, 
                     q = q, 
                     lw = lw.1,
                     runs = runs,
                     Vqn = Vqn,
                     Vqn.sd = Vqn.sd,
                     type.lw = "rnorm",
                     autocorrelation = 3)

## Plot line-point graph with means and standard deviations
plot(NA,
     xlim = c(1, nrow(X.artificial)),
     ylim = c(0.5, q + 1),
     main = "End-member scores with uncertainty")
for(i in 1:q) {
  lines(1:nrow(X.artificial), M$mean[,i] - M$sd[,i] + i, col = i, lty = 2)
  lines(1:nrow(X.artificial), M$mean[,i] + i, col = i, lwd = 2)
  points(1:nrow(X.artificial), M$mean[,i] + i, col = i)
  lines(1:nrow(X.artificial), M$mean[,i] + M$sd[,i] + i, col = i, lty = 2)
}

## EXAMPLE 2
## Calculate Mqs uncertainty
M <- Mqs.uncertainty(X = X.artificial, 
                     q = q, 
                     lw = lw.2,
                     runs = runs,
                     Vqn = Vqn,
                     type.lw = "rnorm")

## Plot point graph with error bars
plot(NA,
     xlim = c(1, nrow(X.artificial)),
     ylim = c(0.5, q + 1),
     main = "End-member scores with uncertainty")
for(i in 1:q) {
  points(1:nrow(X.artificial), M$mean[,i] + i, pch = 3, col = i)
  arrows(1:nrow(X.artificial), M$mean[,i] - M$sd[,i] + i, 
         1:nrow(X.artificial), M$mean[,i] + M$sd[,i] + i, 
         code = 3, angle = 90, length = 0.05, col = i)
}
}

\keyword{EMMA}
