% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DIKUM.R
\name{DIKUM}
\alias{DIKUM}
\title{The discrete Inverted Kumaraswamy family}
\usage{
DIKUM(mu.link = "log", sigma.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "log" link as the default for the sigma.}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a discrete Inverted Kumaraswamy distribution
in the \code{gamlss()} function.
}
\description{
The function \code{DIKUM()} defines the discrete Inverted Kumaraswamy distribution, a two parameter
distribution, for a \code{gamlss.family} object to be used in GAMLSS fitting
using the function \code{gamlss()}.
}
\details{
The discrete Inverted Kumaraswamy distribution with parameters \eqn{\mu} and \eqn{\sigma}
has a support 0, 1, 2, ... and density given by

\eqn{f(x | \mu, \sigma) = (1-(2+x)^{-\mu})^{\sigma}-(1-(1+x)^{-\mu})^{\sigma}}

with \eqn{\mu > 0} and \eqn{\sigma > 0}.

Note: in this implementation we changed the original parameters \eqn{\alpha} and \eqn{\beta}
for \eqn{\mu} and \eqn{\sigma} respectively, we did it to implement this distribution within gamlss framework.
}
\examples{
# Example 1
# Generating some random values with
# known mu and sigma
set.seed(150)
y <- rDIKUM(1000, mu=1, sigma=5)

# Fitting the model
library(gamlss)
mod1 <- gamlss(y ~ 1, sigma.fo = ~1, family=DIKUM,
               control = gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
exp(coef(mod1, what='mu'))
exp(coef(mod1, what='sigma'))

# Example 2
# Generating random values under some model

library(gamlss)

# A function to simulate a data set with Y ~ DIKUM
gendat <- function(n) {
  x1 <- runif(n, min=0.4, max=0.6)
  x2 <- runif(n, min=0.4, max=0.6)
  mu    <- exp(1.21 - 3 * x1) # 0.75 approximately
  sigma <- exp(1.26 - 2 * x2) # 1.30 approximately
  y <- rDIKUM(n=n, mu=mu, sigma=sigma)
  data.frame(y=y, x1=x1, x2=x2)
}

dat <- gendat(n=150)

# Fitting the model
mod2 <- gamlss(y ~ x1, sigma.fo = ~x2, family = "DIKUM", data=dat,
               control=gamlss.control(n.cyc=500, trace=FALSE))

summary(mod2)
}
\references{
\insertRef{EL_Helbawy2022}{DiscreteDists}
}
\seealso{
\link{dDIKUM}.
}
\author{
Daniel Felipe Villa Rengifo, \email{dvilla@unal.edu.co}
}
