#' Complex ID Control
#'
#' @description A compartmental model with several different compartments:  Susceptibles (S), Infected and Pre-symptomatic (P), Infected and Asymptomatic (A), Infected and Symptomatic (I), Recovered and Immune (R) and Dead (D). Also modeled is an environmental pathogen stage (E), and susceptible (Sv) and infected (Iv) vectors.
#'
#' @details Susceptible individuals (S) can become infected by pre-symptomatic (P), asymptomatic (A) or symptomatic (I) hosts. The rates at which infections from the different types of infected individuals (P, A and I) occur are governed by 3 parameters, _b~P~_, _b~A~_, and _b~I~_. Susceptible individuals (S) can also become infected by contact with the environment or infected vectors, at rates _b~E~_ and _b~v~_. Susceptible vectors (S~v~) can become infected by contact with symptomatic hosts at rate _b~h~_. All infected hosts first enter the presymptomatic stage. They remain there for some time (determined by rate _g~P~_, the inverse of which is the average time spent in the presymptomatic stage). A fraction _f_ of presymptomatic hosts move into the asymptomatic category, and the rest become symptomatic infected hosts. Asymptomatic infected hosts recover after some time (specified by the rate _g~A~_). Similarly, the rate _g~I~_ determines the duration the symptomatic hosts stay in the symptomatic state. For symptomatic hosts, two outcomes are possible. Either recovery or death. The parameter _d_ determines the fraction of hosts that die due to disease. Recovered individuals are initially immune to reinfection. They can lose their immunity at rate _w_ and return to the susceptible compartment. Symptomatic and asymptomatic hosts shed pathogen into the environment at rates p~A~ and p~I~. The pathogen in the environment decays at rate _c_. New susceptible hosts and vectors enter the system (are born) at rates _n~h~_ and _n~v~_. Mortality (death unrelated to disease) for hosts and vectors occurs at rates _m~h~_ and _m~v~_.
#'
#' This code was generated by the modelbuilder R package.
#' The model is implemented as a set of ordinary differential equations using the deSolve package.
#' The following R packages need to be loaded for the function to work: deSolve.
#'
#' @param S : starting value for Susceptible : numeric
#' @param P : starting value for Presymptomatic : numeric
#' @param A : starting value for Asymptomatic : numeric
#' @param I : starting value for Symptomatic : numeric
#' @param R : starting value for Recovered : numeric
#' @param D : starting value for Dead : numeric
#' @param E : starting value for Pathogen in Environment : numeric
#' @param Sv : starting value for Susceptible Vector : numeric
#' @param Iv : starting value for Infected Vector : numeric
#' @param nH : Birth rate of susceptible hosts : numeric
#' @param mH : Death rate of hosts : numeric
#' @param nV : Birth rate of susceptible vectors : numeric
#' @param mV : Death rate of vectors : numeric
#' @param bP : rate of transmission from pre-symptomatic hosts : numeric
#' @param bA : rate of transmission from asymptomatic hosts : numeric
#' @param bI : rate of transmission from symptomatic hosts : numeric
#' @param bE : rate of transmission from environment : numeric
#' @param bV : rate of transmission from vectors to hosts : numeric
#' @param bH : rate of transmission to vectors from hosts : numeric
#' @param gP : rate of movement out of P compartment : numeric
#' @param gA : rate of movement out of A compartment : numeric
#' @param gI : rate of movement out of I compartment : numeric
#' @param pI : rate of pathogen shedding into environment by symptomatic hosts : numeric
#' @param pA : rate of pathogen shedding into environment by asymptomatic hosts : numeric
#' @param c : rate of pathogen decay in environment : numeric
#' @param f : fraction of presymptomatic that move to asymptomatic : numeric
#' @param d : fraction of symptomatic infected hosts that die due to disease : numeric
#' @param w : rate of immunity waning : numeric
#' @param tstart : Start time of simulation : numeric
#' @param tfinal : Final time of simulation : numeric
#' @param dt : Time step : numeric
#' @return The function returns the output as a list.
#' The time-series from the simulation is returned as a dataframe saved as list element \code{ts}.
#' The \code{ts} dataframe has one column per compartment/variable. The first column is time.
#' @examples
#' # To run the simulation with default parameters:
#' result <- simulate_Complex_ID_Control_ode()
#' # To choose values other than the standard one, specify them like this:
#' result <- simulate_Complex_ID_Control_ode(P = 2,A = 2,I = 2,R = 0,D = 0,E = 2,Iv = 2)
#' # You can display or further process the result, like this:
#' plot(result$ts[,'time'],result$ts[,'S'],xlab='Time',ylab='Numbers',type='l')
#' print(paste('Max number of S: ',max(result$ts[,'S'])))
#' @section Warning: This function does not perform any error checking. So if you try to do something nonsensical (e.g. have negative values for parameters), the code will likely abort with an error message.
#' @section Model Author: Andreas Handel, Alexis Vittengl
#' @section Model creation date: 2020-09-24
#' @section Code Author: generated by the \code{modelbuilder} R package
#' @section Code creation date: 2021-07-19
#' @export

simulate_Complex_ID_Control_ode <- function(S = 1000, P = 1, A = 1, I = 1, R = 0, D = 0, E = 1, Sv = 1000, Iv = 1, nH = 0.01, mH = 0.001, nV = 0.01, mV = 0.001, bP = 0.02, bA = 0.02, bI = 0.02, bE = 0.02, bV = 0.02, bH = 0.02, gP = 0.7, gA = 0.1, gI = 0.1, pI = 0.04, pA = 0.03, c = 0.7, f = 0.08, d = 0.01, w = 0.003, tstart = 0, tfinal = 100, dt = 0.1)
{
  ##############################
  #Block of ODE equations for deSolve
  ##############################
  Complex_ID_Control_ode_fct <- function(t, y, parms)
  {
    with( as.list(c(y,parms)), { #lets us access variables and parameters stored in y and parms by name
    #StartODES
    #Susceptible : Birth of hosts : Infection by P : Infection by A : Infection by I : Infection by E : Infection by Iv : Waning immunity : Natural death of P :
    dS_mb = +nH -S*bP*P -S*bA*A -S*bI*I -S*bE*E -S*bV*Iv +w*R -mH*S
    #Presymptomatic : Infection by P : Infection by A : Infection by I : Infection by E : Infection by Iv : Presymptomatic moving to asymptomatic : Presymptomatic moving to symptomatic : Natural death of P :
    dP_mb = +S*bP*P +S*bA*A +S*bI*I +S*bE*E +S*bV*Iv -f*gP*P -(1-f)*gP*P -mH*P
    #Asymptomatic : Presymptomatic hosts move to asymptomatic : Asymptomatic recovery : Natural death of A :
    dA_mb = +f*gP*P -gA*A -mH*A
    #Symptomatic : Presymptomatic moving to symptomatic : Symptomatic recovery : Symptomatic death : Natural death of I :
    dI_mb = +(1-f)*gP*P -(1-d)*gI*I -d*gI*I -mH*I
    #Recovered : Asymptomatic recovery : Symptomatic recovery : waning immunity : Natural death of R :
    dR_mb = +gA*A +(1-d)*gI*I -w*R -mH*R
    #Dead : Host death due to infection :
    dD_mb = +d*gI*I
    #Pathogen in Environment : Sheddding by symtpomatic hosts : Sheddding by asymtpomatic hosts : Pathogen decay :
    dE_mb = +pI*I +pA*A -c*E
    #Susceptible Vector : Susceptible vector birth : Infection of vectors : Susceptible vector natural death :
    dSv_mb = +nV -bH*I*Sv -mV*Sv
    #Infected Vector : Infection of vectors : Infected vector natural death  :
    dIv_mb = +bH*I*Sv -mV*Iv
    #EndODES
    list(c(dS_mb,dP_mb,dA_mb,dI_mb,dR_mb,dD_mb,dE_mb,dSv_mb,dIv_mb))
  } ) } #close with statement, end ODE code block

  ##############################
  #Main function code block
  ##############################
  #Creating named vectors
  varvec_mb = c(S = S, P = P, A = A, I = I, R = R, D = D, E = E, Sv = Sv, Iv = Iv)
  parvec_mb = c(nH = nH, mH = mH, nV = nV, mV = mV, bP = bP, bA = bA, bI = bI, bE = bE, bV = bV, bH = bH, gP = gP, gA = gA, gI = gI, pI = pI, pA = pA, c = c, f = f, d = d, w = w)
  timevec_mb = seq(tstart, tfinal,by = dt)
  #Running the model
  simout = deSolve::ode(y = varvec_mb, parms = parvec_mb, times = timevec_mb,  func = Complex_ID_Control_ode_fct, rtol = 1e-12, atol = 1e-12)
  #Setting up empty list and returning result as data frame called ts
  result <- list()
  result$ts <- as.data.frame(simout)
  return(result)
}
